'' =================================================================================================
''
''   File........ jm_i2c.spin
''   Purpose..... Low-level I2C routines (requires pull-ups on SCL and SDA)
''   Cogs Used... 0
''   Author...... Jon "JonnyMac" McPhalen
''                Copyright (c) 2009-2025 Jon McPhalen
''                -- see below for terms of use
''                -- elements inspired by code from Mike Green & Chris Gadd
''                -- see below for terms of use
''   E-mail...... jon.mcphalen@gmail.com
''   Started..... 28 JUL 2009
''   Updated..... 08 AUG 2024
''
'' =================================================================================================

'  IMPORTANT Note: This code requires pull-ups on the SDA _and_ SCL lines -- it does not drive
'  the SCL line high.
'
'  Restored/updated clock stretch feature
'  Added version constant
'  Added block functions


con

  VERSION  = 1_1_2                                              ' 1.1.2


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con

  #0, ACK, NAK


var

  long  scl                                                     ' bus pins
  long  sda


pub null

'' This is not an application


pub setup

'' Setup I2C using Propeller EEPROM pins

  setupx(EE_SCL, EE_SDA)


pub setupx(sclpin, sdapin)

'' Define I2C SCL (clock) and SDA (data) pins

  longmove(@scl, @sclpin, 2)                                    '  copy pins
  dira[scl] := 0                                                '  float to pull-up
  outa[scl] := 0                                                '  write 0 to output reg
  dira[sda] := 0
  outa[sda] := 0

  repeat 9                                                      ' reset bus
    if (ina[sda])
      quit
    dira[scl] := 1
    dira[scl] := 0


pub present(slaveid) : result

'' Pings device, returns true if ACK

  start
  result := (write(slaveid) == ACK)
  stop


pub start

'' Create I2C (re)start sequence

  dira[sda] := 0                                                ' SDA high (float to pull-up)
  dira[scl] := 0                                                ' SCL high
  dira[scl] := 0                                                ' short delay

  dira[sda] := 1                                                ' SDA low
  dira[scl] := 1                                                ' SCL low


pub wait(slaveid) | ackbit

'' Waits for I2C device to be ready for new command
'' -- should not be used before device presence is verified

  repeat
    start
    ackbit := write(slaveid)
  until (ackbit == ACK)


pub stop

'' Create I2C stop sequence

  dira[sda] := 1                                                ' SDA low
  dira[sda] := 1                                                ' delay

  dira[scl] := 0                                                ' SCL high
  dira[sda] := 0                                                ' SDA high


pub write(i2cbyte) : ackbit

'' Write byte to I2C bus
'' -- leaves SCL low

  i2cbyte := !i2cbyte << 24                                     ' prep for msbfirst to dir[]

  repeat 8                                                      ' 8 bits
    dira[sda] := i2cbyte <-= 1                                  ' msb first
    dira[scl] := 0                                              ' SCL high
    repeat while (ina[scl] == 0)                                ' hold (allow clock stretch)
    dira[scl] := 1                                              ' SCL low

  dira[sda] := 0                                                ' relase SDA
  dira[scl] := 0                                                ' SCL high
  repeat while (ina[scl] == 0)                                  ' hold (allow clock stretch)
  ackbit := ina[sda]                                            ' read ack bit
  dira[scl] := 1                                                ' SCL low


pub wr_block(p_src, count) : ackbit

'' Write count bytes from buffer at p_src

  repeat while count
    ackbit |= write(byte[p_src++])


pub read(ackbit) : i2cbyte

'' Read byte from I2C bus

  dira[sda] := 0                                                ' release SDA

  repeat 8
    dira[scl] := 0                                              ' SCL high
    repeat while (ina[scl] == 0)                                ' hold (allow clock stretch)
    i2cbyte := (i2cbyte << 1) | ina[sda]                        ' read the bit
    dira[scl] := 1                                              ' SCL low

  dira[sda] := !ackbit                                          ' output ack bit
  dira[scl] := 0                                                ' SCL high
  repeat while (ina[scl] == 0)                                  ' hold (allow clock stretch)
  dira[scl] := 1                                                ' SCL low


pub rd_block(p_dest, count, ackbit)

'' Read count bytes from bus to p_dest
'' -- all byte ACK'd except last which uses ackbit

  repeat count-1
    byte[p_dest++] := read(ACK)

  byte[p_dest] := read(ackbit)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

}}