'' =================================================================================================
''
''   File........ jm_dual_pwm_demo.spin
''   Purpose.....
''   Cogs Used...
''   Author...... Jon "JonnyMac" McPhalen
''                Copyright (c) 2025 Jon McPhalen
''                -- see below for terms of use
''   E-mail...... jon.mcphalen@gmail.com
''   Started.....
''   Updated..... 04 DEC 2025
''
'' =================================================================================================


con

  VERSION = 0_1_0


con { timing }

  _xinfreq = 5_000_000                                          ' use 5MHz crystal
  _clkmode = xtal1 + pll16x                                     ' x16 = 80MHz

  CLK_FREQ = (_clkmode >> 6) * _xinfreq                         ' system freq as a constant
  MS_001   = CLK_FREQ / 1_000                                   ' ticks in 1ms
  US_001   = CLK_FREQ / 1_000_000                               ' ticks in 1us


con { terminal }

  BR_TERM  = 115_200                                            ' terminal baud rate


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { app io pins }

  PWM1     =  1  { O }
  PWM0     =  0  { O }


con

  #true,  ON, OFF
  #false, NO, YES


obj

' main                                                          ' * master Spin cog
  time : "jm_time_80"                                           '   timing and delays (80MHz)
  io   : "jm_io"                                                '   simple io
  term : "jm_fullduplexserial"                                  ' * serial IO for terminal

' * uses cog when loaded


dat { pre-initialized }


var { globals }


pub main

  setup
' wait_for_terminal(true, 250)

  start_pwm(PWM0, PWM1, 24_000)

  set_duty(0, 20)
  set_duty(1, 95)

  repeat
    waitcnt(0)


pub setup

'' Setup IO and objects for application

  time.start                                                    ' setup timing & delays

  io.start(0, 0)                                                ' clear all pins (master cog)

  term.tstart(BR_TERM)                                          ' start serial for terminal *


pub wait_for_terminal(clear, delay)

'' Wait for terminal to be open and key pressed

  term.rxflush
  term.rx
  if (clear)
    term.tx(term#CLS)
    time.pause(delay #> 0)


con { embedded pwm object }

var

  long  pwmcog
  long  pwmstack[32]

  long  pwmcycle
  long  ch0ticks
  long  ch1ticks


pub start_pwm(pin0, pin1, freq) : result

'' Initialize pwm settings
'' -- pin0 & pin1 are pwm outputs (-1 if channel not used)
'' -- freq is pwm frequency (up to 35kHz)

  stop_pwm

  pwmcycle := clkfreq / (freq <# 35_000)                        ' ticks in pwm period

  pwmcog := cognew(dual_pwm(pin0, pin1), @pwmstack) + 1         ' start the pwm cog

  return pwmcog


pub stop_pwm

  if (pwmcog)
    cogstop(pwmcog-1)
    pwmcog := 0

  longfill(@pwmcycle, 0, 3)


pub set_duty(ch, duty)

'' Set pwm duty cycle for ch, 0..100%

  if (ch == 0) or (ch == 1)
    duty := 0 #> duty <# 100
    ch0ticks[ch] := -pwmcycle * duty / 100


pri dual_pwm(pin0, pin1) | t                                    ' launch with cognew()

'' Run dual-channel pwm
'' -- pin0 and pin1 must be legal
''    * use -1 for pin# if channel not used

  if (pin0 => 0)                                                ' pin0 in use?
    ctra := %00100 << 26 | pin0                                 ' set for pwm/nco mode
    frqa := 1                                                   ' highest resolution
    phsa := 0                                                   ' stop
    dira[pin0] := 1                                             ' make pwm pin output in this cog

  if (pin1 => 0)
    ctrb := %00100 << 26 | pin1
    frqb := 1
    phsb := 0
    dira[pin1] := 1

' pwm loop

  t := cnt                                                      ' sync pwm timing
  repeat
    phsa := ch0ticks                                            ' load pwm ticks
    phsb := ch1ticks
    waitcnt(t += pwmcycle)                                      ' let pwm period expire


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}