'' =================================================================================================
''
''   File....... jm_i2c.spin
''   Purpose.... Low-level I2C routines (requires pull-ups on SCL and SDA)
''   Author..... Jon "JonnyMac" McPhalen
''               Copyright (c) 2009-2022 Jon McPhalen
''               -- see below for terms of use
''               -- elements inspired by code from Mike Green & Chris Gadd
''   E-mail..... jon.mcphalen@gmail.com
''   Started.... 28 JUL 2009
''   Updated.... 01 MAR 2021
''
''   {$P1}
''
'' =================================================================================================

'  IMPORTANT Note: This code requires pull-ups on the SDA _and_ SCL lines -- it does not drive
'  the SCL line high.
'
'  Restored clock stretch feature


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }


con

  #0, ACK, NAK


var

  long  sclpin                                                  ' bus pins
  long  sdapin


pub null

'' This is not an application


pub setup

'' Setup I2C using Propeller EEPROM pins

  setupx(EE_SCL, EE_SDA)


pub setupx(_scl, _sda)

'' Define I2C SCL (clock) and SDA (data) pins

  longmove(@sclpin, @_scl, 2)                                   '  copy pins
  dira[sclpin] := 0                                             '  float to pull-up
  outa[sclpin] := 0                                             '  write 0 to output reg
  dira[sdapin] := 0
  outa[sdapin] := 0

  repeat 9
    if (ina[sdapin])
      quit                                                     ' reset device
    dira[sclpin] := 1
    dira[sclpin] := 0


pub present(slaveid)

'' Pings device, returns true if ACK

  start

  return (write(slaveid) == ACK)


pub wait(slaveid) | ackbit

'' Waits for I2C device to be ready for new command

  repeat
    start
    ackbit := write(slaveid)
  until (ackbit == ACK)


pub start

'' Create I2C start sequence
'' -- will wait if I2C bus SCL pin is held low

  dira[sdapin] := 0                                             ' float SDA (1)
  dira[sclpin] := 0                                             ' float SCL (1)

  dira[sdapin] := 1                                             ' SDA low (0)
  dira[sclpin] := 1                                             ' SCL low (0)


pub write(i2cbyte) | ackbit

'' Write byte to I2C bus
'' -- leaves SCL low

  i2cbyte := !i2cbyte << 24                                     ' move msb (bit7) to bit31

  repeat 8                                                      ' output eight bits
    dira[sdapin] := i2cbyte <-= 1                               ' send msb first
    dira[sclpin] := 0                                           ' SCL high (float to p/u)
    dira[sclpin] := 1                                           ' SCL low

  dira[sdapin] := 0                                             ' relase SDA to read ack bit
  dira[sclpin] := 0                                             ' SCL high (float to p/u)
  repeat while (ina[sclpin] == 0)                               ' hold for clock stretch
  ackbit := ina[sdapin]                                         ' read ack bit
  dira[sclpin] := 1                                             ' SCL low

  return ackbit


pub read(ackbit) : i2cbyte

'' Read byte from I2C bus

  dira[sdapin] := 0                                             ' make sdapin input

  repeat 8
    dira[sclpin] := 0                                           ' SCL high (float to p/u)
    i2cbyte := (i2cbyte << 1) | ina[sdapin]                     ' read the bit
    dira[sclpin] := 1                                           ' SCL low

  dira[sdapin] := !ackbit                                       ' output ack bit
  dira[sclpin] := 0                                             ' clock it
  repeat while (ina[sclpin] == 0)                               ' hold for clock stretch
  dira[sclpin] := 1

  return i2cbyte


pub stop

'' Create I2C stop sequence

  dira[sdapin] := 1                                             ' SDA low
  dira[sclpin] := 0                                             ' float SCL
  dira[sdapin] := 0                                             ' float SDA


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

}}