''***************************************
''  File....... 4_encoder_8_buttons.spin
''  Purpose.... 4 Encoder + 8 Button driver via 74HC165 shift registers
''  Author..... Based on JonnyMac's 74HC165 and Mark Tillotson's encoder patterns
''  Started.... 2026-01-20
''  Updated.... 2026-02-15
''
''  PASM reads from HUB before updating - allows external clamping
''***************************************

con { fixed io pins }

  RX1      = 31  { I }
  TX1      = 30  { O }
  SDA1     = 29  { I/O }
  SCL1     = 28  { I/O }

var

  long  cog
  long  din
  long  clk
  long  load
  long  latch_mode
  long  enc1, enc2, enc3, enc4
  long  err1, err2, err3, err4
  long  buttons
  long  buttons_latched

pub null

pub start(dpin, cpin, lpin, enable_latch) : okay

  stop
  longmove(@din, @dpin, 3)
  latch_mode := enable_latch
  longfill(@enc1, 0, 10)
  okay := cog := cognew(@entry, @din) + 1

pub stop

  if cog
    cogstop(cog - 1)
    cog := 0

pub get_encoder(encoder_num) : value

  case encoder_num
    1: value := enc1
    2: value := enc2
    3: value := enc3
    4: value := enc4
    other: value := 0

pub get_error(encoder_num) : value

  case encoder_num
    1: value := err1
    2: value := err2
    3: value := err3
    4: value := err4
    other: value := 0

pub set_encoder(encoder_num, value)

  case encoder_num
    1: enc1 := value
    2: enc2 := value
    3: enc3 := value
    4: enc4 := value

pub reset_encoder(encoder_num)

  set_encoder(encoder_num, 0)

pub get_encoder_base : ptr

  ptr := @enc1

pub get_buttons : btnstate

  btnstate := buttons & $FF

pub get_button(button_num) : btnval

  if (button_num => 0) and (button_num =< 7)
    btnval := (buttons >> button_num) & 1
  else
    btnval := 0

pub get_buttons_latched : latchstate

  latchstate := buttons_latched & $FF
  buttons_latched := 0

pub get_button_latched(button_num) : latchval

  if (button_num => 0) and (button_num =< 7)
    latchval := (buttons_latched >> button_num) & 1
    buttons_latched &= !(1 << button_num)
  else
    latchval := 0

dat

                        org     0

entry
                        mov     t1, par

                        rdlong  t2, t1
                        mov     din_mask, #1
                        shl     din_mask, t2

                        add     t1, #4
                        rdlong  t2, t1
                        mov     clk_mask, #1
                        shl     clk_mask, t2

                        add     t1, #4
                        rdlong  t2, t1
                        mov     load_mask, #1
                        shl     load_mask, t2

                        add     t1, #4
                        rdlong  latch_en, t1

                        add     t1, #4
                        mov     enc1_ptr, t1
                        add     t1, #4
                        mov     enc2_ptr, t1
                        add     t1, #4
                        mov     enc3_ptr, t1
                        add     t1, #4
                        mov     enc4_ptr, t1
                        add     t1, #4
                        mov     err1_ptr, t1
                        add     t1, #4
                        mov     err2_ptr, t1
                        add     t1, #4
                        mov     err3_ptr, t1
                        add     t1, #4
                        mov     err4_ptr, t1
                        add     t1, #4
                        mov     btn_ptr, t1
                        add     t1, #4
                        mov     btnl_ptr, t1

                        andn    dira, din_mask
                        or      dira, clk_mask
                        andn    outa, clk_mask
                        or      dira, load_mask
                        or      outa, load_mask

                        mov     error0, #0
                        mov     error1, #0
                        mov     error2, #0
                        mov     error3, #0
                        mov     btn_old, #0
                        mov     stable_count, #0

                        call    #read_165
                        mov     state, enc_bits
                        mov     last_stable, enc_bits

main_loop
                        call    #read_165

                        mov     newstate, enc_bits

                        cmp     newstate, state wz
              if_z      jmp     #:same_state

                        mov     stable_count, #0
                        mov     state, newstate
                        jmp     #:check_buttons

:same_state
                        add     stable_count, #1
                        cmp     stable_count, debounce_limit wc, wz
              if_b      jmp     #:check_buttons

                        cmp     newstate, last_stable wz
              if_z      jmp     #:check_buttons

                        mov     change, last_stable
                        xor     change, newstate

                        ' Process encoder 0 (bits 0-1) = Encoder 1
                        test    change, enc0_mask wc, wz
              if_z      jmp     #:next1
              if_nc     add     error0, #1
              if_nc     wrlong  error0, err1_ptr
              if_nc     jmp     #:next1

                        rdlong  count0, enc1_ptr         ' READ from HUB first
                        test    newstate, enc0_mask wc
              if_nc     xor     change, enc0_mask
                        test    change, enc0_mlow wz
              if_nz     add     count0, #1
              if_z      sub     count0, #1
                        wrlong  count0, enc1_ptr

:next1
                        test    change, enc1_mask wc, wz
              if_z      jmp     #:next2
              if_nc     add     error1, #1
              if_nc     wrlong  error1, err2_ptr
              if_nc     jmp     #:next2

                        rdlong  count1, enc2_ptr         ' READ from HUB first
                        test    newstate, enc1_mask wc
              if_nc     xor     change, enc1_mask
                        test    change, enc1_mlow wz
              if_nz     add     count1, #1
              if_z      sub     count1, #1
                        wrlong  count1, enc2_ptr

:next2
                        test    change, enc2_mask wc, wz
              if_z      jmp     #:next3
              if_nc     add     error2, #1
              if_nc     wrlong  error2, err3_ptr
              if_nc     jmp     #:next3

                        rdlong  count2, enc3_ptr         ' READ from HUB first
                        test    newstate, enc2_mask wc
              if_nc     xor     change, enc2_mask
                        test    change, enc2_mlow wz
              if_nz     add     count2, #1
              if_z      sub     count2, #1
                        wrlong  count2, enc3_ptr

:next3
                        test    change, enc3_mask wc, wz
              if_z      jmp     #:done_enc
              if_nc     add     error3, #1
              if_nc     wrlong  error3, err4_ptr
              if_nc     jmp     #:done_enc

                        rdlong  count3, enc4_ptr         ' READ from HUB first
                        test    newstate, enc3_mask wc
              if_nc     xor     change, enc3_mask
                        test    change, enc3_mlow wz
              if_nz     add     count3, #1
              if_z      sub     count3, #1
                        wrlong  count3, enc4_ptr

:done_enc
                        mov     last_stable, newstate

:check_buttons
                        cmp     latch_en, #0 wz
              if_z      jmp     #:skip_latch

                        mov     t1, btn_new
                        andn    t1, btn_old
                        rdlong  t2, btnl_ptr
                        or      t2, t1
                        wrlong  t2, btnl_ptr

:skip_latch
                        mov     btn_old, btn_new

                        jmp     #main_loop

read_165
                        andn    outa, load_mask
                        nop                         ' Stretch load pulse
                        or      outa, load_mask

                        mov     shift_in, #0
                        mov     bit_cnt, #16

:read_loop
                        shl     shift_in, #1
                        test    din_mask, ina wc
                        muxc    shift_in, #1
                        or      outa, clk_mask
                        nop                      'slow clock
                        andn    outa, clk_mask
                        nop                      'slow clock
                        djnz    bit_cnt, #:read_loop

                        mov     btn_new, shift_in
                        shr     btn_new, #8
                        and     btn_new, #$FF
                        wrlong  btn_new, btn_ptr

                        mov     enc_bits, shift_in
                        and     enc_bits, #$FF

read_165_ret            ret

debounce_limit          long    4

enc0_mask               long    %00000011
enc0_mlow               long    %00000001
enc1_mask               long    %00001100
enc1_mlow               long    %00000100
enc2_mask               long    %00110000
enc2_mlow               long    %00010000
enc3_mask               long    %11000000
enc3_mlow               long    %01000000

t1                      long    0
t2                      long    0

din_mask                long    0
clk_mask                long    0
load_mask               long    0
latch_en                long    0

enc1_ptr                long    0
enc2_ptr                long    0
enc3_ptr                long    0
enc4_ptr                long    0
err1_ptr                long    0
err2_ptr                long    0
err3_ptr                long    0
err4_ptr                long    0
btn_ptr                 long    0
btnl_ptr                long    0

shift_in                long    0
bit_cnt                 long    0
enc_bits                long    0

count0                  long    0
count1                  long    0
count2                  long    0
count3                  long    0

error0                  long    0
error1                  long    0
error2                  long    0
error3                  long    0

state                   long    0
newstate                long    0
change                  long    0
last_stable             long    0
stable_count            long    0

btn_new                 long    0
btn_old                 long    0

                        fit     496

con

{{
  Terms of Use: MIT License
}}