'' =================================================================================================
''
''   File........ jm_p1-ee_explorer_v060.spin
''   Purpose..... Explore and modify P1 boot EEPROM (32K or 64K)
''   Author...... Jon "JonnyMac" McPhalen
''                Copyright (c) 2026 Jon McPhalen
''                -- see below for terms of use
''   E-mail...... jon.mcphalen@gmail.com
''   Started.....
''   Updated..... 26 FEB 2026
''
'' =================================================================================================

{
  Works with
  * Propeller Tool w/+ PST
  * Spin Tools 0.53.1 w/ internal terminal (in PST mode)
  * FlexProp w/ internal terminal when compiled to PASM
    -- error in output with bytecode mode as of FlexProp 7.6.1


  * * *  W A R N I N G !  * * *

  Modifying EE addresses in the program space (below DBASE) could lead to program
  corruption. Use Write feature with caution.

}


con

  VERSION = 0_6_0


con { timing }

  _xinfreq = 5_000_000                                          ' use 5MHz crystal
  _clkmode = xtal1 + pll16x                                     ' x16 = 80MHz

  CLK_FREQ = (_clkmode >> 6) * _xinfreq                         ' system freq as a constant
  MS_001   = CLK_FREQ / 1_000                                   ' ticks in 1ms
  US_001   = CLK_FREQ / 1_000_000                               ' ticks in 1us


con { terminal }

  BR_TERM  = 115_200                                            ' terminal baud rate


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { app io pins }


con

  #true,  ON, OFF
  #false, NO, YES

  EE_SPD = 1_000_000                                           ' i2c clock speed

  EE_32K = 32 * 1024
  EE_64K = 64 * 1024

  LINES  = 8                                                   ' show 8 lines at a time
  DELTA  = LINES * 16                                          ' address delta for display

  ESC    = 27


obj

' main                                                          ' * master Spin cog
  time : "jm_time_80"                                           '   timing and delays (80MHz)
  io   : "jm_io"                                                '   simple io
  ee   : "jm_24xx256_fast"                                      '   EE access
  term : "jm_fullduplexserial"                                  ' * serial IO for terminal

' * uses cog when loaded


dat { menus }

  s_Banner      byte    "P1 EEPROM Explorer v0.0.0", 13
                byte    0

  s_Menu1       byte    "----------------", 13
                byte    "P1 EEPROM Size  ", 13
                byte    "----------------", 13
                byte    "[3] 32K (24x256)", 13
                byte    "[6] 64K (24x512)", 13
                byte    0

  s_Menu2       byte    "-----------------", 13
                byte    "P1 EEPROM Actions", 13
                byte    "-----------------", 13
                byte    "[S]can           ", 13
                byte    "[E]rase/Fill     ", 13
                byte    0

  s_Menu3a      byte    "------------------------", 13
                byte    "P1 EEPROM Erase/Fill    ", 13
                byte    "------------------------", 13
                byte    "[1] Erase low 32K ($FF) ", 13
                byte    "[2] Fill  low 32K ($00) ", 13
                byte    0

  s_Menu3b      byte    "[3] Erase high 32K ($FF)", 13
                byte    "[4] Fill  high 32K ($00)", 13
                byte    0

  s_EscMenu     byte    13
                byte    "[Esc]", 13
                byte    0


var

  long  eesize
  long  eeaddr

' initialization structure (16 bytes)

  long  pgmfreq                                                 ' clkfreq
  byte  clockmode                                               ' clock mode bits
' byte  checksum                                                ' image checksum
  word  pbase                                                   ' start of program ($0010)
  word  vbase                                                   ' start of variables
  word  dbase                                                   ' start of data (stack)
' word  pcurr                                                   ' current program location
' word  dcurr                                                   ' current data (stack) location

  byte  linebuf[16]
  byte  ascbuf[17]


pub main | x

  setup
  wait_for_terminal(true, 250)

{
  repeat
    term.fstr0(string("Enter value: "))
    x := enter_value
    term.tx(13)
    term.fstr3(string("-- d:%d    h:$%x    b:\%%b\r\r"), x, x, x)
    time.pause(100)

  repeat
}

  repeat
    eesize := get_ee_size
    run_user_action


pub get_ee_size : k

'' Set EE size for 32K or 64K

  term.rxflush

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu1)

    k := term.rx
    if (k == "3")
      return EE_32K
    if (k == "6")
      return EE_64K


pub run_user_action | k

'' Run Scan, Erase, or escape to size menu

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu2)
    term.str(@s_EscMenu)

    term.rxflush
    k := term.rx
    if (k == "s") or (k == "S")
      run_scan
    elseif (k == "e") or (k == "E")
      run_erase_fill
    elseif (k == ESC)
      return


pub run_scan

'' Scan and display contents of EEPROM

  s_Banner[20] := "0" + VERSION  / 100
  s_Banner[22] := "0" + VERSION  / 10 // 10
  s_Banner[24] := "0" + VERSION // 10

  term.tx(term#CLS)
  term.str(@s_Banner)
  term.fstr1(string("-- %dK bytes "), eesize/1024)
  if (eesize == EE_64K)
    term.str(string("(BOOT + 32K)"))
  term.txn(13, 2)

  show_pgm_stats                                                ' details about pgm in lower ee

  eeaddr := $0000

  repeat
    show_block                                                  ' show block at eeaddr
    if (get_command == ESC)                                     ' process user input
      quit


pub show_pgm_stats

'' Show initialization data if present

  ee.rd_block($0000, 16, @linebuf)                              ' initialization data

  bytemove(@pgmfreq, @linebuf[0], 4)                            ' extract clkfreq
  if (pgmfreq < 20_000) or (pgmfreq > 100_000_000)              ' if out of norms
    return                                                      '  don't show pgm stats

  clockmode := linebuf[4]
  bytemove(@pbase, @linebuf[$06], 2)
  bytemove(@vbase, @linebuf[$08], 2)
  bytemove(@dbase, @linebuf[$0A], 2)

  term.str(string("Program Stats", 13))
  term.fstr1(string("-- CLKFREQ... %d\r"), pgmfreq)
  term.fstr2(string("-- CLKMODE... \%%.8b  %s\r"), clockmode, clock_mode_str(clockmode))
  term.fstr2(string("-- PBASE..... $%.4x (%4d longs)\r"),   pbase, (vbase-pbase)>>2)
  term.fstr2(string("-- VBASE..... $%.4x (%4d longs)\r"),   vbase, (dbase-8-vbase)>>2)
  term.fstr2(string("-- DBASE..... $%.4x (%4d longs)\r\r"), dbase, ($8000-dbase+8)>>2)


pub clock_mode_str(cm) : p_str

  case cm
    %0_0_0_00_000 : return string("RCFAST")
    %0_0_0_00_001 : return string("RCSLOW")

    %0_0_1_00_010 : return string("XINPUT")
    %0_1_1_00_011 : return string("XINPUT + PLL1X ")
    %0_1_1_00_100 : return string("XINPUT + PLL2X ")
    %0_1_1_00_101 : return string("XINPUT + PLL4X ")
    %0_1_1_00_110 : return string("XINPUT + PLL8X ")
    %0_1_1_00_111 : return string("XINPUT + PLL16X")

    %0_0_1_01_010 : return string("XTAL1")
    %0_1_1_01_011 : return string("XTAL1 + PLL1X ")
    %0_1_1_01_100 : return string("XTAL1 + PLL2X ")
    %0_1_1_01_101 : return string("XTAL1 + PLL4X ")
    %0_1_1_01_110 : return string("XTAL1 + PLL8X ")
    %0_1_1_01_111 : return string("XTAL1 + PLL16X")

    %0_0_1_10_010 : return string("XTAL2")
    %0_1_1_10_011 : return string("XTAL2 + PLL1X ")
    %0_1_1_10_100 : return string("XTAL2 + PLL2X ")
    %0_1_1_10_101 : return string("XTAL2 + PLL4X ")
    %0_1_1_10_110 : return string("XTAL2 + PLL8X ")
    %0_1_1_10_111 : return string("XTAL2 + PLL16X")

    %0_0_1_11_010 : return string("XTAL3")
    %0_1_1_11_011 : return string("XTAL3 + PLL1X ")
    %0_1_1_11_100 : return string("XTAL3 + PLL2X ")
    %0_1_1_11_101 : return string("XTAL3 + PLL4X ")
    %0_1_1_11_110 : return string("XTAL3 + PLL8X ")
    %0_1_1_11_111 : return string("XTAL3 + PLL16X")

    other         : return string("?")


pub show_block | i, b

  repeat LINES
    ee.rd_block(eeaddr, 16, @linebuf)                           ' get line from EE
    if (eeaddr < $8000)
      term.tx("L")
    else
      term.tx("H")
    term.fstr1(string(" $%.4x : "), eeaddr)                      ' show the address
    repeat i from 0 to 15                                       ' display line bytes
      b := linebuf[i]
      term.fstr1(string("%.2x "), b)                            ' as hex byte
      if ((i & %11) == %11)
        term.tx(" ")                                            ' pad between longs
      if (b => " ") and (b =< 127)                              ' if ascii
        ascbuf[i] := b                                          '  add to string
      else
        ascbuf[i] := "."                                        '  else sub with dot
    ascbuf[16] := 0
    term.fstr1(string("  %s\r"), @ascbuf)
    eeaddr += 16                                                ' update addr for this line
    eeaddr //= eesize                                           ' keep address within size

  term.tx(13)                                                   ' pad for next block


dat { strings }

  s_CmdMenu     byte    "Cmd: [?] [n] [p] [l] [h] [g] [v] [w] [Esc]", 13
                byte    13
                byte    0

  s_Help        byte    "h | H | ?            Show help", 13
                byte    "n | N | Enter        Next block", 13
                byte    "p | P | Backspace    Previous block", 13
                byte    "l | L                Go to $0000", 13
                byte    "h | H                Go to $8000", 13
                byte    "g | G                Go to address", 13
                byte    "v | V                Get value(s) from address", 13
                byte    "w | W                Write to [aligned] address", 13
                byte    "Esc                  Back to previous menu", 13
                byte    13
                byte    0

  s_SizeError   byte    "ERROR: [H] not valid. EEPROM size is 32K", 13
                byte    13
                byte    0

  s_AddrError   byte    13
                byte    "-- ERROR: Invalid address", 13
                byte    13
                byte    0


pub get_command : k

  term.str(@s_CmdMenu)

  repeat
    term.rxflush
    k := term.rx
    case k
      "?" :
        show_help

      13, "n", "N" :                                            ' [Enter]
        quit

      08, "p", "P" :                                            ' [Backspace]
        eeaddr -= DELTA * 2                                     ' back to block before last displayed
        if (eeaddr < 0)
          eeaddr += eesize                                      ' fix underflow
        quit

      "l", "L" :
        eeaddr := $0000                                         ' start of lower ee
        quit

      "h", "H" :
        if (eesize == EE_64K)
          eeaddr := $8000                                       ' start of upper ee
        else
          term.str(@s_SizeError)
          last_block
        quit

      "g", "G" :
        cmd_goto                                                ' go to block with user address
        quit

      "v", "V" :
        cmd_values                                              ' show value(s) of user address
        last_block
        quit

      "w", "W" :
        cmd_write                                               ' write value to user address
        quit

      27 :                                                      ' [Esc]
        quit


pub last_block

'' Sets global EE address to last block displayed

  eeaddr -= DELTA
  if (eeaddr < 0)
    eeaddr += eesize


pub cmd_goto | check, addr

  term.str(string("<GOTO> Address: "))

  check := enter_value(@addr)

  if (check == 0) or (addr < $0000) or (addr => eesize)
    term.str(@s_AddrError)
    last_block                                                  ' repeat last block displayed
  else
    term.txn(13, 2)
    eeaddr := get_block_start(addr)                             ' set to block with that address


pub cmd_values | check, addr, val

  term.str(string("<VALUE(S)> Address: "))

  check := enter_value(@addr)

  if (check == 0)
    return

  if (addr < 0) or (addr => eesize)
    term.str(@s_AddrError)
    return

  ee.rd_xblock(addr, 4, @val)

  term.fstr1(string("Value(s): B = %d  "), val.byte[0])

  if ((addr & %01) == $00)                                      ' on word boundary?
    term.fstr1(string("W = %d  "), val.word[0])

  if ((addr & %11) == $00)                                      ' on long boundary?
    term.fstr1(string("L = %d  "), val)

  term.txn(13, 2)


pub cmd_write | check, addr, size, value

'' Allow user to write to specified location
'' -- word and long values must be aligned per P1 memory layout

  term.str(string("<WRITE> Address: "))

  check := enter_value(@addr)

  if (check == 0) or (addr < $0000) or (addr => eesize)
    term.str(@s_AddrError)
    last_block
    return

  term.str(string("  Size [b|w|l]: "))

  term.rxflush
  size := ucase(term.rx)
  if (size <> "B") and (size <> "W") and (size <> "L")
    term.str(string(13, "-- ERROR: Invalid size", 13, 13))
    last_block
    return

' validate alignment for words and longs

  if (size == "W")
    if (addr & %1 <> %0)
      term.str(string(13, "-- ERROR: Address is not word-aligned", 13, 13))
      last_block
      return

  if (size == "L")
    if (addr & %11 <> %00)
      term.str(string(13, "-- ERROR: Address is not long-aligned", 13, 13))
      last_block
      return

  term.fstr1(string("%c  Value: "), size)
  check := enter_value(@value)

  if (check == 0)
    term.str(string(13, "-- ERROR: No value entered", 13, 13))
    last_block
    return

  if (size == "B")
    ee.wr_byte(addr, value.byte[0])

  elseif (size == "W")
    ee.wr_word(addr, value.word[0])

  else (size == "L")
    ee.wr_long(addr, value)

  eeaddr := get_block_start(addr)
  term.txn(13, 2)


pub enter_value(p_value) : len | value, base, sign, limit, k

'' Allow user to enter decimal, hex, or binary value
'' -- hex and binary required indicators ($, %) as first char

  value := 0
  base  := 10
  sign  := 1
  limit := 10

  repeat
    term.rxflush
    k := ucase(term.rx)
    case k
      "$" :
        if (len == 0)                                           ' must be first char
          len++
          base := 16
          limit := 9
          value := 0
          term.tx(k)

      "%" :
        if (len == 0)                                           ' must be first char
          len++
          base := 2
          limit := 33
          value := 0
          term.tx(k)

      "-" :
        if (len == 0)                                           ' must be first char
          len++
          base := 10
          sign := -1
          limit := 11
          value := 0
          term.tx(k)

      "0".."1" :                                                ' all bases
        if (len < limit)
          len++
          value := (value * base) + (k - "0")
          term.tx(k)

      "2".."9" :                                                ' dec and hex
        if (len < limit)
          len++
          if (base == 10)
            value := (value * base) + (k - "0")
          else
            value := (value << 4) + (k - "0")
          term.tx(k)

      "A".."F" :                                                ' hex only
        if (base == 16) and (len < 9)
          len++
          value := (value << 4) + (k - "A" + 10)
          term.tx(k)

      08 :                                                      ' [Backspace]
        if (len > 0)
          len--
          value /= base
          if (len == 0)
            base := 10
            sign := 1
            limit := 10
          term.tx(8)
          term.tx(" ")                                          ' for FlexProp terminal                                                                                                                                                                ' for FlexProp terminal
          term.tx(8)                                            '

      13 :                                                      ' [Enter]
        if (base == 10)
          value *= sign
        else
          len -= 1                                              ' remove indicator

        long[p_value] := value
        return len


pub xget_address : value | base, idx, k

'' Enter desired EE address in decimal or hex format
'' -- simple editor; leading zeroes count in decimal mode

  term.str(string("Addr: "))

  base := 10
  idx  := 0

  repeat
    term.rxflush
    k := ucase(term.rx)
    case k
      "$" :
        if (idx == 0)                                           ' must be first char
          idx++
          base := 16
          value := 0
          term.tx(k)

      "0".."9" :                                                ' dec or hex
        if (idx < 5)
          idx++
          value := (value * base) + (k - "0")
          term.tx(k)

      "A".."F" :                                                ' hex only
        if (base == 16) and (idx < 5)
          idx++
          value := (value << 4) + (k - "A" + 10)
          term.tx(k)

      08 :                                                      ' [Backspace]
        if (idx > 0)
          idx--
          value /= base
          if (idx == 0)
            base := 10
          term.tx(8)
          term.tx(" ")                                          ' for FlexProp terminal                                                                                                                                                                ' for FlexProp terminal
          term.tx(8)                                            '

      13 :                                                      ' [Enter]
        if (base == 10)
          term.fstr1(string(" ($%.4x)"), value)                 ' show addr as hex
        term.txn(13, 2)                                         ' pad for next block
        return value


pri ucase(c) : uc

'' Return uppercase version c
'' -- only affects alphas

  if (c => "a") and (c =< "z")
    c &= %11011111

  return c


pub get_block_start(addr) : bstart

'' Return start of block that contains addr

  if (addr =< 0)
    return $0000

  if (addr => eesize)
    return eesize - DELTA

  if (addr // DELTA == 0)
    return addr

  return (addr / DELTA) * DELTA


pub run_erase_fill : k

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu3a)
    if (eesize == EE_64K)
      term.str(@s_Menu3b)
    term.str(@s_EscMenu)

    term.rxflush
    k := term.rx
    case k
      "1" : fill_low($FF)                                       ' erase
      "2" : fill_low($00)                                       ' clear (like Spin compilers)
      "3" :
        if (eesize == EE_64K)
          fill_high($FF)
      "4" :
        if (eesize == EE_64K)
          fill_high($00)
      27  : return


pri fill_low(b) | ofs

  term.tx(term#CLS)
  if (b == $FF)
    term.str(string("Erasing "))
  else
    term.str(string("Filling "))
  term.str(string("lower 32K ($0000..$7FFF)", 13))

  ofs := $0000
  repeat while (ofs < $8000)
    ee.fill(ofs, ee#PG_SIZE, b)
    ofs += ee#PG_SIZE


pri fill_high(b) | ofs

  term.tx(term#CLS)
  if (b == $FF)
    term.str(string("Erasing "))
  else
    term.str(string("Filling "))
  term.str(string("upper 32K ($8000..$FFFF)", 13))

  ofs := $8000
  repeat while (ofs < $1_0000)
    ee.fill(ofs, ee#PG_SIZE, b)
    ofs += ee#PG_SIZE



pub show_help

  term.str(@s_Help)


pub setup

'' Setup IO and objects for application

  time.start                                                    ' setup timing & delays

  io.start(0, 0)                                                ' clear all pins (master cog)

  ee.start(%000, EE_SPD)                                        ' connect to EE (fast)

  term.tstart(BR_TERM)                                          ' start serial for terminal *


pub wait_for_terminal(clear, mshold)

'' Wait for terminal to be open and key pressed

  term.rxflush
  term.rx
  if (clear)
    term.tx(term#CLS)
    time.pause(mshold #> 0)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}