'' =================================================================================================
''
''   File........ jm_p1-ee_explorer_v041.spin
''   Purpose..... Explore P1 boot EEPROM (32K or 64K)
''   Author...... Jon "JonnyMac" McPhalen
''                Copyright (c) 2026 Jon McPhalen
''                -- see below for terms of use
''   E-mail...... jon.mcphalen@gmail.com
''   Started.....
''   Updated..... 18 FEB 2026
''
'' =================================================================================================

{

  If using Spin Tools there is a problem with the internal terminal when using the
  get address method during the scan/display. Marco has confirmed this is and is sorting
  it out for a future release. If you don't need to specify a specific address, there
  is no problem. If you want to use that feature PST works fine (Windows only).

  If using FlexProp, do not use bytecode mode. I don't know why but the formatted string
  methods don't seem to work with the FlexProp bytecode mode.

}


con

  VERSION = 0_4_1                                               ' small edit for FlexProp terminal


con { timing }

  _xinfreq = 5_000_000                                          ' use 5MHz crystal
  _clkmode = xtal1 + pll16x                                     ' x16 = 80MHz

  CLK_FREQ = (_clkmode >> 6) * _xinfreq                         ' system freq as a constant
  MS_001   = CLK_FREQ / 1_000                                   ' ticks in 1ms
  US_001   = CLK_FREQ / 1_000_000                               ' ticks in 1us


con { terminal }

  BR_TERM  = 115_200                                            ' terminal baud rate


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { app io pins }


con

  #true,  ON, OFF
  #false, NO, YES

  EE_SPD = 1_000_000                                           ' i2c clock speed

  EE_32K = 32 * 1024
  EE_64K = 64 * 1024

' lines should be power-of-2
' -- each line is 16 bytes

  LINES  = 8                                                   ' show 8 lines at a time
  DELTA  = LINES * 16                                          ' address delta for display

  ESC    = 27


obj

' main                                                          ' * master Spin cog
  time : "jm_time_80"                                           '   timing and delays (80MHz)
  io   : "jm_io"                                                '   simple io
  ee   : "jm_24xx256_fast"                                      '   EE access
  term : "jm_fullduplexserial"                                  ' * serial IO for terminal

' * uses cog when loaded


dat { menus }

  s_Banner      byte    "P1 EEPROM Explorer", 13
                byte    0

  s_Menu1       byte    "----------------", 13
                byte    "P1 EEPROM Size  ", 13
                byte    "----------------", 13
                byte    "[3] 32K (24x256)", 13
                byte    "[6] 64K (24x512)", 13
                byte    0

  s_Menu2       byte    "-----------------", 13
                byte    "P1 EEPROM Actions", 13
                byte    "-----------------", 13
                byte    "[S]can           ", 13
                byte    "[E]rase/Fill     ", 13
                byte    0

  s_Menu3a      byte    "------------------------", 13
                byte    "P1 EEPROM Erase/Fill    ", 13
                byte    "------------------------", 13
                byte    "[1] Erase low 32K ($FF) ", 13
                byte    "[2] Fill  low 32K ($00) ", 13
                byte    0

  s_Menu3b      byte    "[3] Erase high 32K ($FF)", 13
                byte    "[4] Fill  high 32K ($00)", 13
                byte    0

  s_EscMenu     byte    13
                byte    "[Esc]", 13
                byte    0


var

  long  eesize
  long  eeaddr

  long  pgmfreq
  byte  clockmode
' byte  checksum
  word  pbase
  word  vbase
  word  dbase

  byte  linebuf[16]
  byte  ascbuf[17]


pub main

  setup
  wait_for_terminal(true, 250)

  repeat
    eesize := get_ee_size
    run_user_action


pub get_ee_size : k

'' Set EE size for 32K or 64K

  term.rxflush

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu1)

    k := term.rx
    if (k == "3")
      return EE_32K
    if (k == "6")
      return EE_64K


pub run_user_action | k

'' Run Scan, Erase, or escape to size menu

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu2)
    term.str(@s_EscMenu)

    term.rxflush
    k := term.rx
    if (k == "s") or (k == "S")
      run_scan
    elseif (k == "e") or (k == "E")
      run_erase_fill
    elseif (k == ESC)
      return


pub run_scan

'' Scan and display contents of EEPROM

  term.tx(term#CLS)
  term.str(@s_Banner)
  term.fstr1(string("-- %dK bytes "), eesize/1024)
  if (eesize == EE_64K)
    term.str(string("(BOOT + 32K)"))
  term.txn(13, 2)

  show_pgm_stats

  eeaddr := $0000

  repeat
    show_lines
    if (get_command == ESC)
      quit


pub show_pgm_stats

'' Show initialization data if present

  ee.rd_block($0000, 16, @linebuf)                              ' initialization data

  bytemove(@pgmfreq, @linebuf[0], 4)                            ' extract clkfreq
  if (pgmfreq < 20_000) or (pgmfreq > 100_000_000)              ' if out of norms
    return                                                      '  don't show pgm stats

  clockmode := linebuf[4]
  bytemove(@pbase, @linebuf[$06], 2)
  bytemove(@vbase, @linebuf[$08], 2)
  bytemove(@dbase, @linebuf[$0A], 2)

  term.str(string("Program Stats", 13))
  term.fstr1(string("-- CLKFREQ... %d\r"), pgmfreq)
  term.fstr1(string("-- CLKMODE... \%%.8b\r"), clockmode)
  term.fstr2(string("-- PBASE..... $%.4x (%4d longs)\r"),   pbase, (vbase-pbase)>>2)
  term.fstr2(string("-- VBASE..... $%.4x (%4d longs)\r"),   vbase, (dbase-8-vbase)>>2)
  term.fstr2(string("-- DBASE..... $%.4x (%4d longs)\r\r"), dbase, ($8000-dbase+8)>>2)


pub show_lines | i, b

  repeat LINES
    ee.rd_block(eeaddr, 16, @linebuf)                           ' get line from EE
    term.fstr1(string("$%.4x : "), eeaddr)                      ' show the address
    repeat i from 0 to 15                                       ' display line bytes
      b := linebuf[i]
      term.fstr1(string("%.2x "), b)                            ' as hex byte
      if ((i & %11) == %11)
        term.tx(" ")                                            ' pad between longs
      if (b => " ") and (b =< 127)                              ' if ascii
        ascbuf[i] := b                                          '  add to string
      else
        ascbuf[i] := "."                                        '  else sub with dot
    ascbuf[16] := 0
    term.fstr1(string("  %s\r"), @ascbuf)
    eeaddr += 16                                                ' update addr for this line
    eeaddr //= eesize                                           ' keep address within size


dat

  s_CmdMenu     byte    13
                byte    "[N]ext  [P]revious  [L]ow  [H]igh  [G]oto  [V]alue  [Esc]", 13
                byte    13
                byte    0

  s_SizeError   byte    "Error : [H] not valid. EEPROM size is 32K", 13
                byte    13
                byte    0

  s_Address     byte    "Address: ", 0

  s_AddrError   byte    "Error : Invalid address for this EEPROM", 13
                byte    13
                byte    0


pub get_command : k | addr

  term.str(@s_CmdMenu)

  repeat
    term.rxflush
    k := term.rx
    case k
      13, "n", "N" :                                            ' [Enter]
        quit

      08, "p", "P" :                                            ' [Backspace]
        eeaddr -= DELTA * 2                                     ' back to block before last displayed
        if (eeaddr < 0)
          eeaddr += eesize                                      ' fix underflow
        quit

      "l", "L" :
        eeaddr := $0000                                         ' start of lower ee
        quit

      "h", "H" :
        if (eesize == EE_64K)
          eeaddr := $8000                                       ' start of upper ee
        else
          term.str(@s_SizeError)
          eeaddr -= DELTA                                       ' display last block
          if (eeaddr < 0)
            eeaddr += eesize
        quit

      "g", "G" :
        addr := get_address                                     ' allow user address
        eeaddr := get_block_start(addr)                         ' set to block with that address
        quit

      "v", "V" :
        addr := get_address                                     ' allow user address
        show_values(addr)                                       ' if valid show byte, word, long
        eeaddr -= DELTA                                         ' redisplay last block
        if (eeaddr < 0)
          eeaddr += eesize
        quit

      27 :                                                      ' [Esc]
        quit


pub get_address : value | base, idx, k

'' Enter desired EE address in decimal or hex format
'' -- simple editor; leading zeroes count in decimal mode

  term.str(string("Address : "))

  base := 10
  idx  := 0

  repeat
    term.rxflush
    k := ucase(term.rx)
    case k
      "$" :
        if (idx == 0)                                           ' must be first char
          idx++
          base := 16
          value := 0
          term.tx(k)

      "0".."9" :                                                ' dec or hex
        if (idx < 5)
          idx++
          value := (value * base) + (k - "0")
          term.tx(k)

      "A".."F" :                                                ' hex only
        if (base == 16) and (idx < 5)
          idx++
          value := (value << 4) + (k - "A" + 10)
          term.tx(k)

      08 :                                                      ' [Backspace]
        if (idx > 0)
          idx--
          value /= base
          if (idx == 0)
            base := 10
          term.tx(8)
          term.tx(" ")                                          ' for FlexProp terminal                                                                                                                                                                ' for FlexProp terminal
          term.tx(8)                                            '

      13 :                                                      ' [Enter]
        if (base == 10)
          term.fstr1(string(" ($%.4x)"), value)                 ' show addr as hex
        term.txn(13, 2)                                         ' pad for next block
        return value


pri ucase(c) : uc

'' Return uppercase version c
'' -- only affects alphas

  if (c => "a") and (c =< "z")
    c &= %11011111

  return c


pub get_block_start(addr) : bstart

'' Return start of block that contains addr

  if (addr =< 0)
    return $0000

  if (addr => eesize)
    return eesize - DELTA

  return (addr / DELTA) * DELTA


pub show_values(addr) : value

  if (addr => eesize)
    term.str(@s_AddrError)
    return

  ee.rd_xblock(addr, 4, @value)

  term.fstr1(string("Value(s): B = %d  "), value.byte[0])

  if ((addr & %01) == $00)                                      ' on word boundary?
    term.fstr1(string("W = %d  "), value.word[0])

  if ((addr & %11) == $00)
    term.fstr1(string("L = %d  "), value)                       ' on long boundary?

  term.txn(13, 2)


pub run_erase_fill : k

  repeat
    term.tx(term#CLS)
    term.str(@s_Menu3a)
    if (eesize == EE_64K)
      term.str(@s_Menu3b)
    term.str(@s_EscMenu)

    term.rxflush
    k := term.rx
    case k
      "1" : fill_low($FF)                                       ' erase
      "2" : fill_low($00)                                       ' clear (like Spin compilers)
      "3" :
        if (eesize == EE_64K)
          fill_high($FF)
      "4" :
        if (eesize == EE_64K)
          fill_high($00)
      27  : return


pri fill_low(b) | ofs

  term.tx(term#CLS)
  if (b == $FF)
    term.str(string("Erasing "))
  else
    term.str(string("Filling "))
  term.str(string("lower 32K ($0000..$7FFF)", 13))

  ofs := $0000
  repeat while (ofs < $8000)
    ee.fill(ofs, ee#PG_SIZE, b)
    ofs += ee#PG_SIZE


pri fill_high(b) | ofs

  term.tx(term#CLS)
  if (b == $FF)
    term.str(string("Erasing "))
  else
    term.str(string("Filling "))
  term.str(string("upper 32K ($8000..$FFFF)", 13))

  ofs := $8000
  repeat while (ofs < $1_0000)
    ee.fill(ofs, ee#PG_SIZE, b)
    ofs += ee#PG_SIZE


pub setup

'' Setup IO and objects for application

  time.start                                                    ' setup timing & delays

  io.start(0, 0)                                                ' clear all pins (master cog)

  ee.start(%000, EE_SPD)                                        ' connect to EE (fast)

  term.tstart(BR_TERM)                                          ' start serial for terminal *


pub wait_for_terminal(clear, mshold)

'' Wait for terminal to be open and key pressed

  term.rxflush
  term.rx
  if (clear)
    term.tx(term#CLS)
    time.pause(mshold #> 0)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}
