'' =================================================================================================
''
''   File....... jm_24xx256_fast.spin
''   Purpose.... R/W routines for 24xx256 (32K) EEPROM
''   Author..... Jon "JonnyMac" McPhalen
''               Copyright (c) 2009-2026 Jon McPhalen
''               -- see below for terms of use
''   E-mail..... jon.mcphalen@gmail.com
''   Started....
''   Updated.... 18 FEB 2026
''
'' =================================================================================================


con

  VERSION  = 1_0_0


con { fixed io pins }

  RX_PGM   = 31  { I }                                          ' serial / programming
  TX_PGM   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED27    = 27  { O }                                          ' LEDs on PAB and FLiP
  LED26    = 26  { O }


con { i2c address }

  EE_WRITE = %1010_000_0
  EE_READ  = %1010_000_1

  EE_SIZE  = 32 * 1024
  PG_SIZE  = 64
  PAGES    = EE_SIZE / PG_SIZE
  LAST_PG  = PAGES - 1

  ACK      = i2c#ACK
  NAK      = i2c#NAK


obj

  i2c : "jm_i2c_fast"


var

  long  devid


pub null

'' This is not an application

  repeat
    waitcnt(0)


pub start(device, hz) : result

'' Setup 24xx256 on boot eeprom bus
'' -- device is the device address %000 - %111
''    * %000 is boot eeprom

  result := startx(EE_SCL, EE_SDA, device, hz)


pub startx(sclpin, sdapin, device, hz) : result

'' Setup 24xx256 on user-defined bus
'' -- sclpin and sdapin are connections to I2C bus
'' -- device is the device address %000 - %111
''    * %000 is boot eeprom

  stop                                                          ' stop fast i2c cog if running

  dira[sclpin] := 0                                             ' float bus pins in this cog
  dira[sdapin] := 0

  result := i2c.setupx(sclpin, sdapin, hz)                      ' start i2c

  if (result)                                                   ' if started
    devid := EE_WRITE | ((%000 #> device <# %111) << 1)         ' setup device id


pub stop

'' Stops fast I2C interface

  i2c.terminate
  devid := 0


pub wr_byte(addr, b)

'' Write byte to eeprom

  return wr_block(addr, 1, @b)


pub wr_word(addr, w)

'' Write word to eeprom

  return wr_block(addr, 2, @w)


pub wr_long(addr, l)

'' Write long to eeprom

  return wr_block(addr, 4, @l)


pub wr_block(addr, n, p_src) : ackbit

'' Write block of n bytes from p_src to eeprom
'' -- be mindful of address/page size in device to prevent page wrap-around

  i2c.wait(devid)
  i2c.write(addr.byte[1])                                       ' msb of address
  i2c.write(addr.byte[0])                                       ' lsb of address
  ackbit := i2c.wr_block(p_src, n)                              ' * fast block write
  i2c.stop


pub wr_xblock(addr, n, p_src) : ackbit

'' Write block of n bytes from p_src to eeprom
'' -- re-addresses each byte to allow page boundary crossing
'' -- slower than wr_block()

  ackbit := i2c#ACK                                             ' assume okay

  repeat n
    i2c.wait(devid)
    i2c.write(addr.byte[1])                                     ' msb of address
    i2c.write(addr.byte[0])                                     ' lsb of address
    ackbit |= i2c.write(byte[p_src++])                          ' write a byte
    i2c.stop
    addr++                                                      ' inc address


pub wr_str(addr, p_str) : ackbit | b

'' Write z-string at p_str to eeprom
'' -- string can cross page boundary

  ackbit := i2c#ACK                                             ' assume okay

  repeat
    b := byte[p_str++]                                          ' get byte from string
    ackbit |= wr_byte(addr++, b)                                ' write
    if (b == 0)                                                 ' end of string?
      quit                                                      '  yes, we're done


pub fill(addr, n, b) : ackbit

'' Write byte b to eeprom n times, starting with addr
'' -- be mindful of address/page size in device to prevent page wrap-around

  i2c.wait(devid)
  i2c.write(addr.byte[1])                                       ' msb of address
  i2c.write(addr.byte[0])                                       ' lsb of address
  ackbit := i2c#ACK                                             ' assume okay
  repeat n
    ackbit |= i2c.write(b)                                      ' write the byte
  i2c.stop


pub rd_byte(addr) | b

'' Return byte value from eeprom

  rd_block(addr, 1, @b)

  return b & $FF                                                ' clean-up { local var }


pub rd_word(addr) | w

'' Return word value eeprom

  rd_block(addr, 2, @w)

  return w & $FFFF


pub rd_long(addr) : l

'' Return long value from eeprom

  rd_block(addr, 4, @l)

  return l


pub rd_block(addr, n, p_dest)

'' Read block of n bytes from eeprom to address at p_dest
'' -- be mindful of address/page size in device to prevent page wrap-around

  i2c.wait(devid)
  i2c.write(addr.byte[1])                                       ' msb of address
  i2c.write(addr.byte[0])                                       ' lsb of address
  i2c.start                                                     ' restart for read
  i2c.write(devid | $01)                                        ' device read
  i2c.rd_block(p_dest, n, NAK)                                  ' fast block read
  i2c.stop


pub rd_xblock(addr, n, p_dest)

'' Read block of n bytes from eeprom to address at p_dest
'' -- block can cross page boundary

  repeat n
    byte[p_dest++] := rd_byte(addr++)                           ' move from EE to p_dest


pub rd_str(addr, p_dest) | b

'' Read (arbitrary-length) z-string, store at p_dest
'' -- string can cross page boundary

  repeat
    b := rd_byte(addr++)                                        ' read byte from device
    byte[p_dest++] := b                                         ' write to destination
    if (b == 0)                                                 ' at end?
      quit                                                      '  if yes, we're done


pub page_num(addr)

'' Returns page # of addr

  return (addr / PG_SIZE)


pub page_ok(addr, len) | pg0, pg1

'' Returns true if len bytes will fit into current page

  pg0 := page_num(addr)
  pg1 := page_num(addr + len-1)

  return (pg1 == pg0)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}