'' =================================================================================================
''
''   File....... jm_ssd1306_128x32-i2c.spin
''   Purpose....
''   Author..... Jon McPhalen
''               Copyright (c) 2016-2026 Jon McPhalen
''               -- see below for terms of use
''   E-mail..... jon.mcphalen@gmail.com
''   Started....
''   Updated.... 25 OCT 2025
''
'' =================================================================================================


con

  VERSION = 0_2_1


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }


con { ssd1306 settings }

  DISP_X = 128                                                  ' 128 pixels wide
  DISP_Y =  32                                                  '  64 pixels tall
  DISP_R = DISP_Y / 8                                           ' display rows

  DISP_BYTES = DISP_X * DISP_R                                  ' bytes in display

  X_BIT  = 1 << 7 { 128 }
  X_MASK = $7F

  OLED_ID   = %0111_100_0                                       ' 128x32 slave id

  CMD  = $00
  DATA = $40

  #0, P_OFF, P_ON, P_XOR                                        ' pixel styles

  #1, F_NORM, F_WIDE, F_TALL, F_BIG                            ' font sizes


con

  #0, OFF, ON
  #0, NO, YES


obj

  i2c : "jm_i2c_fast"


var

  long  bufidx
  byte  buffer[DISP_BYTES]


pub null

'' This is not an application


pub start

'' Start SSD1306 driver on EE pins

  startx(EE_SCL, EE_SDA)


pub startx(sclpin, sdapin) | ms1, pntr, len, t

'' Start SSD1306 driver
'' -- I2C may be on any two pins
''    * add 10K pull-ups if none on external device(s)

  ms1 := clkfreq / 1000

  i2c.setupx(sclpin, sdapin, 400_000)                           ' run at 400kHz

  pntr := @Init1306
  len  := @EOS - pntr

  t := cnt
  repeat len
    write(byte[pntr++], CMD)
    waitcnt(t += ms1)

  clear


dat { initialization sequence }

  Init1306      byte    $AE                                     ' display off
                byte    $D5, $80                                ' set display clock divide ratio/oscillator frequency
                byte    $A8, $1F                                ' set multiplex ratio (32-1)
                byte    $D3, $00                                ' set display offset
                byte    $40                                     ' set start line address
                byte    $8D, $14                                ' charge pump setting (enable)
                byte    $20, $00                                ' memory addressing mode (horizontal)
                byte    $A1                                     ' set segment re-map (column 127 mapped to SEG0)
                byte    $C8                                     ' set COM output scan direction (remapped)
                byte    $DA, $02                                ' set COM pins hardware configuration
                byte    $81, $8F                                ' set contrast control
                byte    $D9, $F1                                ' set pre-charge period
                byte    $DB, $40                                ' set VCOMH deselect level
                byte    $A4                                     ' entire display ON (resume to RAM content)
                byte    $A6                                     ' set normal display (not inverted)
                byte    $2E                                     ' deactivate scroll
                byte    $AF                                     ' display ON

  EOS           byte    $FF                                     ' End of Sequence


pub enable(state)

'' Enables/disables display

  if (state)
    write($AF, CMD)
  else
    write($AE, CMD)


pub flip(state)

'' Flip screen around horizontal axis

  if (state)
    write($A0, CMD)                                        ' seg remap
    write($C0, CMD)                                        ' C0 (flipped [rotated 180])
  else
    write($A1, CMD)                                        ' seg remap
    write($C8, CMD)                                        ' C8 (not flipped)


pub invert(state)

'' Invert pixels screen pixels

  if (state)
    write($A7, CMD)
  else
    write($A6, CMD)


pub clear

'' Clears display

  write($21, CMD)                                               ' set column address
  write($00, CMD)                                               ' first column
  write($7F, CMD)                                               ' last column

  write($22, CMD)                                               ' set page address
  write($00, CMD)                                               ' first page
  write($03, CMD)                                               ' last page

  i2c.start
  i2c.write(OLED_ID)
  i2c.write(DATA)
  repeat DISP_BYTES
    i2c.write($00)
  i2c.stop

  bytefill(@buffer, $00, DISP_BYTES)

  goto_xy(0, 0)
  bufidx := 0


pub clear_row(y)

  goto_xy(0, y)

  i2c.start
  i2c.write(OLED_ID)
  i2c.write(DATA)
  repeat DISP_X
    i2c.write($00)
  i2c.stop

  bytefill(@buffer[y * DISP_X], $00, DISP_X)

  goto_xy(0, y)


pub home

'' Moves cursor to upper/left coordinate of display

  goto_xy(0, 0)


pub crsr_xy(x, y)

'' Moves character cursor to x, y
'' -- x is column (0..15)
'' -- y is row (0..3)

  goto_xy(x << 3, y)


pub goto_xy(x, y)

'' Moves display cursor to x, y
'' -- x is column (0..127)
'' -- y is row (0..3)

  write($B0 |  y,       CMD)                                    ' move to row
  write($00 | (x & $F), CMD)                                    ' set low column on row
  write($10 | (x >> 4), CMD)                                    ' set high column on row

  bufidx := (y * DISP_X) + x                                    ' update buffer index


pub goto_idx(idx)

'' Moves cursor to index in display buffer
'' -- idx is 0 (upper left) to 511 (lower right)

  goto_xy(idx & $7F, idx >> 7)


pub putc(c, size, style) | p_map, cx, cy, i, tc

'' Write character c to OLED
'' -- c is ASCII char from $20..$7F
'' -- size is 1 (normal), 2 (wide), 3 (tall), or 4 (big)
'' -- style is 0 (off), 1 (normal), 2 (xor w/bkg)

  if ((c < $20) or (c > $7F))                                   ' limit to ASCII set
    return

  p_map := @Ascii + ((c - $20) << 3)                            ' pointer to character map

  if ((size == F_NORM) or (size == F_WIDE))                     ' 8x8 or 16x8 font
    repeat 8
      c := byte[p_map++]
      case style
        P_OFF : c ^= $FF
        P_XOR : c ^= buffer[bufidx]
      repeat size
        write(c, DATA)

  elseif ((size == F_TALL) or (size == F_BIG))                  ' 8x16 or 16x16 font
    cx := bufidx & $7F                                          ' save upper left of char
    cy := bufidx >> 7

    repeat 8
      c := byte[p_map++]
      tc := 0                                                   ' clear tall column
      repeat i from 0 to 7                                      ' double pixels in column
        if (c & |<i)
          tc |= %11 << (i << 1)
      case style
        P_OFF : tc ^= $FFFF
        P_XOR : tc.byte[0] ^= buffer[bufidx]
                tc.byte[1] ^= buffer[bufidx+DISP_X]
      repeat size-2
        write(tc.byte[0], DATA)
        goto_xy(cx, cy+1)
        write(tc.byte[1], DATA)
        goto_xy(++cx, cy)


pub str(p_str, size, style)

'' Write z-string (at p_str) to LCD
'' -- set position first

  repeat strsize(p_str)
    putc(byte[p_str++], size, style)


pub l_str(p_str, size, style, y)

'' Left justify text on line y (0..3)

  goto_xy(0, y)
  str(p_str, size, style)


pub r_str(p_str, size, style, y)

'' Right justify text on line y (0..3)

  if (size == F_NORM)
    goto_xy((16 - strsize(p_str)) << 3, y)
  else
    goto_xy((08 - strsize(p_str)) << 4, y)

  str(p_str, size, style)


pub c_str(p_str, size, style, y)

'' Center text on line y (0..3)

  if (size == F_NORM) or (size == F_TALL)
    goto_xy((16 - strsize(p_str)) << 2, y)
  elseif (size == F_WIDE) or (size == F_BIG)
    goto_xy((08 - strsize(p_str)) << 3, y)
  else
    return

  str(p_str, size, style)


pub dec(value, size, style) : zf | nf, d, c

'' Print a signed decimal number

  if (value < 0)
    putc("-", size, style)
    if (value == negx)
      value := posx
      nf := 1
    else
      value := -value
      nf := 0

  d := 1_000_000_000

  repeat 10
    c := "0" + (value / d)
    if (value => d) or (zf) or (d == 1)
      if (d == 1) and (nf)
        c := "8"
      putc(c, size, style)
      zf := true
      value //= d
    d /= 10


pub rjdec(value, width, pchar, size, style) | tmpval, pad

'' Print right-justified decimal value
'' -- val is value to print
'' -- width is width of (padded) field for value
'' -- pchar is [leading] pad character (usually "0" or " ")

'  Original code by Dave Hein
'  -- modifications by Jon McPhalen

  if (value => 0)                                               ' if positive
    tmpval := value                                             '  copy value
    pad := width - 1                                            '  make room for 1 digit
  else
    if (value == negx)                                          '  if max negative
      tmpval := posx                                            '    use max positive for width
    else                                                        '  else
      tmpval := -value                                          '    make positive
    pad := width - 2                                            '  make room for sign and 1 digit

  repeat while (tmpval => 10)                                   ' adjust pad for value width > 1
    pad--
    tmpval /= 10

  repeat pad                                                    ' print pad
    putc(pchar, size, style)

  dec(value, size, style)                                       ' print value


pub hex(value, digits, size, style)

'' Print a hexadecimal number

  value <<= (8 - digits) << 2
  repeat digits
    putc(lookupz((value <-= 4) & $F : "0".."9", "A".."F"), size, style)


pub bin(value, digits, size, style)

'' Print a binary number

  value <<= 32 - digits
  repeat digits
    putc((value <-= 1) & 1 + "0", size, style)


pub pixel(x, y, mode) | idx, ofs, d

'' Paint pixel at x, y
'' -- x coordinate is 0..127
'' -- y coordinate is 0..63

  if ((x < 0) or (x => DISP_X))                                 ' validate
    return

  if ((y < 0) or (y => DISP_Y))
    return

  idx := ((y >> 3) << 7) + x                                    ' index into buffer
  ofs := y & %111                                               ' bit offset for pixel

  d := buffer[idx]                                              ' get column byte from buffer

  case mode                                                     ' update pixel
    P_OFF:
      d &= !(1 << ofs)

    P_ON:
      d |= (1 << ofs)

    P_XOR:
      d ^= (1 << ofs)

    other:
      return

  buffer[idx] := d                                              ' update buffer

  goto_xy(x, y >> 3)

  write(d, DATA)                                                ' update lcd


pub line(x1, y1, x2, y2, mode) | dx, dy, stepx, stepy, frac

'' Draw line between x1/y1 and x2/y2

  dy := y2 - y1
  dx := x2 - x1

  if (dx < 0)
    dx := -dx
    stepx := -1
  else
    stepx := 1

  if (dy < 0)
    dy := -dy
    stepy := -1
  else
    stepy := 1

  dx <<= 1
  dy <<= 1

  pixel(x1, y1, mode)

  if (dx > dy)
    frac := dy - (dx >> 1)
    repeat while (x1 <> x2)
      if (frac => 0)
        y1 += stepy
        frac -= dx
      x1 += stepx
      frac += dy
      pixel(x1, y1, mode)

  else
    frac := dx - (dy >> 1)
    repeat while (y1 <> y2)
      if (frac => 0)
        x1 += stepx
        frac -= dy
      else
        y1 += stepy
        frac += dx
        pixel(x1, y1, mode)


pub box(x1, y1, x2, y2, mode)

  line(x1, y1, x2, y1, mode)
  line(x2, y1, x2, y2, mode)
  line(x2, y2, x1, y2, mode)
  line(x1, y2, x1, y1, mode)


pub box_fill(x1, y1, x2, y2, mode) | yy

  repeat yy from y1 to y2
    line(x1, yy, x2, yy, mode)


pub write(b, cdbit)

'' Write raw byte to LCD

  i2c.start
  i2c.write(OLED_ID)
  i2c.write(cdbit)
  i2c.write(b)
  i2c.stop


dat { 8x8 font }

  Ascii         byte    $00, $00, $00, $00, $00, $00, $00, $00   ' 20 space
                byte    $00, $00, $5f, $00, $00, $00, $00, $00   ' 21 !
                byte    $00, $07, $00, $07, $00, $00, $00, $00   ' 22 "
                byte    $00, $14, $7f, $14, $7f, $14, $00, $00   ' 23 #
                byte    $00, $24, $2a, $7f, $2a, $12, $00, $00   ' 24 $
                byte    $00, $23, $13, $08, $64, $62, $00, $00   ' 25 %
                byte    $00, $36, $49, $55, $22, $50, $00, $00   ' 26 &
                byte    $00, $00, $05, $03, $00, $00, $00, $00   ' 27 '
                byte    $00, $00, $1c, $22, $41, $00, $00, $00   ' 28 (
                byte    $00, $41, $22, $1c, $00, $00, $00, $00   ' 29 )
                byte    $00, $14, $08, $3e, $08, $14, $00, $00   ' 2a *
                byte    $00, $08, $08, $3e, $08, $08, $00, $00   ' 2b +
                byte    $00, $00, $50, $30, $00, $00, $00, $00   ' 2c ,
                byte    $00, $08, $08, $08, $08, $08, $00, $00   ' 2d -
                byte    $00, $00, $60, $60, $00, $00, $00, $00   ' 2e .
                byte    $00, $20, $10, $08, $04, $02, $00, $00   ' 2f /
                byte    $00, $3e, $51, $49, $45, $3e, $00, $00   ' 30 0
                byte    $00, $00, $42, $7f, $40, $00, $00, $00   ' 31 1
                byte    $00, $42, $61, $51, $49, $46, $00, $00   ' 32 2
                byte    $00, $21, $41, $45, $4b, $31, $00, $00   ' 33 3
                byte    $00, $18, $14, $12, $7f, $10, $00, $00   ' 34 4
                byte    $00, $27, $45, $45, $45, $39, $00, $00   ' 35 5
                byte    $00, $3c, $4a, $49, $49, $30, $00, $00   ' 36 6
                byte    $00, $01, $71, $09, $05, $03, $00, $00   ' 37 7
                byte    $00, $36, $49, $49, $49, $36, $00, $00   ' 38 8
                byte    $00, $06, $49, $49, $29, $1e, $00, $00   ' 39 9
                byte    $00, $00, $36, $36, $00, $00, $00, $00   ' 3a :
                byte    $00, $00, $56, $36, $00, $00, $00, $00   ' 3b ;
                byte    $00, $08, $14, $22, $41, $00, $00, $00   ' 3c <
                byte    $00, $14, $14, $14, $14, $14, $00, $00   ' 3d =
                byte    $00, $00, $41, $22, $14, $08, $00, $00   ' 3e >
                byte    $00, $02, $01, $51, $09, $06, $00, $00   ' 3f ?
                byte    $00, $32, $49, $79, $41, $3e, $00, $00   ' 40 @
                byte    $00, $7e, $11, $11, $11, $7e, $00, $00   ' 41 A
                byte    $00, $7f, $49, $49, $49, $36, $00, $00   ' 42 B
                byte    $00, $3e, $41, $41, $41, $22, $00, $00   ' 43 C
                byte    $00, $7f, $41, $41, $22, $1c, $00, $00   ' 44 D
                byte    $00, $7f, $49, $49, $49, $41, $00, $00   ' 45 E
                byte    $00, $7f, $09, $09, $09, $01, $00, $00   ' 46 F
                byte    $00, $3e, $41, $49, $49, $7a, $00, $00   ' 47 G
                byte    $00, $7f, $08, $08, $08, $7f, $00, $00   ' 48 H
                byte    $00, $00, $41, $7f, $41, $00, $00, $00   ' 49 I
                byte    $00, $20, $40, $41, $3f, $01, $00, $00   ' 4a J
                byte    $00, $7f, $08, $14, $22, $41, $00, $00   ' 4b K
                byte    $00, $7f, $40, $40, $40, $40, $00, $00   ' 4c L
                byte    $00, $7f, $02, $0c, $02, $7f, $00, $00   ' 4d M
                byte    $00, $7f, $04, $08, $10, $7f, $00, $00   ' 4e N
                byte    $00, $3e, $41, $41, $41, $3e, $00, $00   ' 4f O
                byte    $00, $7f, $09, $09, $09, $06, $00, $00   ' 50 P
                byte    $00, $3e, $41, $51, $21, $5e, $00, $00   ' 51 Q
                byte    $00, $7f, $09, $19, $29, $46, $00, $00   ' 52 R
                byte    $00, $46, $49, $49, $49, $31, $00, $00   ' 53 S
                byte    $00, $01, $01, $7f, $01, $01, $00, $00   ' 54 T
                byte    $00, $3f, $40, $40, $40, $3f, $00, $00   ' 55 U
                byte    $00, $1f, $20, $40, $20, $1f, $00, $00   ' 56 V
                byte    $00, $3f, $40, $38, $40, $3f, $00, $00   ' 57 W
                byte    $00, $63, $14, $08, $14, $63, $00, $00   ' 58 X
                byte    $00, $07, $08, $70, $08, $07, $00, $00   ' 59 Y
                byte    $00, $61, $51, $49, $45, $43, $00, $00   ' 5a Z
                byte    $00, $00, $7f, $41, $41, $00, $00, $00   ' 5b [
                byte    $00, $02, $04, $08, $10, $20, $00, $00   ' 5c 
                byte    $00, $00, $41, $41, $7f, $00, $00, $00   ' 5d ]
                byte    $00, $04, $02, $01, $02, $04, $00, $00   ' 5e ^
                byte    $00, $40, $40, $40, $40, $40, $00, $00   ' 5f _
                byte    $00, $00, $01, $02, $04, $00, $00, $00   ' 60 `
                byte    $00, $20, $54, $54, $54, $78, $00, $00   ' 61 a
                byte    $00, $7f, $48, $44, $44, $38, $00, $00   ' 62 b
                byte    $00, $38, $44, $44, $44, $20, $00, $00   ' 63 c
                byte    $00, $38, $44, $44, $48, $7f, $00, $00   ' 64 d
                byte    $00, $38, $54, $54, $54, $18, $00, $00   ' 65 e
                byte    $00, $08, $7e, $09, $01, $02, $00, $00   ' 66 f
                byte    $00, $0c, $52, $52, $52, $3e, $00, $00   ' 67 g
                byte    $00, $7f, $08, $04, $04, $78, $00, $00   ' 68 h
                byte    $00, $00, $44, $7d, $40, $00, $00, $00   ' 69 i
                byte    $00, $20, $40, $44, $3d, $00, $00, $00   ' 6a j
                byte    $00, $7f, $10, $28, $44, $00, $00, $00   ' 6b k
                byte    $00, $00, $41, $7f, $40, $00, $00, $00   ' 6c l
                byte    $00, $7c, $04, $18, $04, $78, $00, $00   ' 6d m
                byte    $00, $7c, $08, $04, $04, $78, $00, $00   ' 6e n
                byte    $00, $38, $44, $44, $44, $38, $00, $00   ' 6f o
                byte    $00, $7c, $14, $14, $14, $08, $00, $00   ' 70 p
                byte    $00, $08, $14, $14, $18, $7c, $00, $00   ' 71 q
                byte    $00, $7c, $08, $04, $04, $08, $00, $00   ' 72 r
                byte    $00, $48, $54, $54, $54, $20, $00, $00   ' 73 s
                byte    $00, $04, $3f, $44, $40, $20, $00, $00   ' 74 t
                byte    $00, $3c, $40, $40, $20, $7c, $00, $00   ' 75 u
                byte    $00, $1c, $20, $40, $20, $1c, $00, $00   ' 76 v
                byte    $00, $3c, $40, $30, $40, $3c, $00, $00   ' 77 w
                byte    $00, $44, $28, $10, $28, $44, $00, $00   ' 78 x
                byte    $00, $0c, $50, $50, $50, $3c, $00, $00   ' 79 y
                byte    $00, $44, $64, $54, $4c, $44, $00, $00   ' 7a z
                byte    $00, $00, $08, $36, $41, $00, $00, $00   ' 7b {
                byte    $00, $00, $00, $7f, $00, $00, $00, $00   ' 7c |
                byte    $00, $00, $41, $36, $08, $00, $00, $00   ' 7d }
                byte    $00, $10, $08, $08, $10, $08, $00, $00   ' 7e ~
                byte    $00, $78, $46, $41, $46, $78, $00, $00   ' 7f


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}