'' =================================================================================================
''
''   File....... jm_i2c_fast.spin
''   Purpose.... Low-level I2C routines (requires pull-ups on SCL and SDA)
''   Author..... Jon "JonnyMac" McPhalen
''               Copyright (c) 2009-2026 on McPhalen
''               -- elements inspired by code from Mike Green
''   E-mail.....
''   Started.... 28 JUL 2009
''   Updated.... 17 APR 2024
''
'' =================================================================================================

'  IMPORTANT Note: This code requires pull-ups on the SDA _and_ SCL lines -- it does not drive
'  the SCL line high.
'
'  Cog value stored in DAT table which is shared across all object uses; all objects that use
'  this object MUST use the same I2C bus pins
'  -- updated to allow cog var to support multiple instances/pins if desired
'     * if cog is var, mutliple instances are okay
'     * if cog is in dat, only one instance allowed
'
'  17 APR 2024 Update
'  -- setup() methods do not return to caller until background cog is running
'
'  22 JUL 2018 Update
'  -- added wr_byte(), wr_word(), and wr_long methods
'  -- renamed bwrite() to wr_block()
'  -- added rd_byte(), rd_word(), and rd_long methods
'  -- renamed bread() to rd_block()


con { fixed io pins }

  RX_PGM   = 31  { I }                                          ' serial / programming
  TX_PGM   = 30  { O }

  SDA_EE   = 29  { I/O }                                        ' i2c / eeprom
  SCL_EE   = 28  { I/O }


con

  #0, ACK, NAK

  #1, I2C_START, I2C_WRITE, I2C_READ, I2C_STOP                  ' byte


var

  long  i2ccmd
  long  i2cparam1                                               ' for future use
  long  i2cparam2                                               ' for future use
  long  i2cresult

  long  cog                                                     ' allow multiple instances (cogs)


dat

' cog           long    0                                       ' only 1 instance (cog)


pub null

'' This is not an application


pub setup(hz) : result

'' Start i2c cog on propeller i2c bus
'' -- aborts if cog already running
'' -- example: i2c.setup(400_000)

  if (cog)
    return cog

  return setupx(SCL_EE, SDA_EE, hz)                             ' use propeller i2c pins


pub setupx(sclpin, sdapin, hz) : result

'' Start i2c cog on any set of pins
'' -- aborts if cog already running
'' -- example: i2c.setupx(SCL, SDA, 400_000)

  if (cog)
    return cog

  i2ccmd.byte[0] := sclpin                                      ' setup pins
  i2ccmd.byte[1] := sdapin
  i2ccmd.word[1] := clkfreq / hz                                ' ticks in full cycle

  cog := cognew(@fast_i2c, @i2ccmd) + 1                         ' start the cog

  if (cog)
    repeat while (i2ccmd <> 0)                                  ' wait for cog to be ready

  return cog


pub terminate

'' Kill i2c cog

  if (cog)
    cogstop(cog-1)
    cog := 0

  longfill(@i2ccmd, 0, 4)


pub present(ctrl) : result | tmp

'' Pings device, returns true it ACK

  i2ccmd := I2C_START
  repeat while (i2ccmd <> 0)

  tmp.byte[0] := I2C_WRITE                                      ' build packed command
  tmp.byte[1] := 1
  tmp.word[1] := @ctrl

  i2ccmd := tmp
  repeat while (i2ccmd <> 0)

  return (i2cresult == ACK)


pub wait(ctrl)

'' Waits for I2C device to be ready for new command

  repeat
    if (present(ctrl))
      quit


pub start

'' Create I2C start sequence
'' -- will wait if I2C bus SDA pin is held low

  i2ccmd := I2C_START
  repeat while (i2ccmd <> 0)


pub write(b) : acknak

'' Write byte to I2C bus

  return wr_block(@b, 1)


pub wr_byte(b) : acknak

'' Write byte to I2C bus

  return wr_block(@b, 1)


pub wr_word(w) : acknak

'' Write word to I2C bus
'' -- Little Endian

  return wr_block(@w, 2)


pub wr_long(l) : acknak

'' Write long to I2C bus
'' -- Little Endian

  return wr_block(@l, 4)


pub wr_block(p_src, count) : acknak | cmd

'' Write block of count bytes from p_src to I2C bus
'' -- destination block should be continuous in same ee page

  cmd.byte[0] := I2C_WRITE
  cmd.byte[1] := count
  cmd.word[1] := p_src

  i2ccmd := cmd
  repeat while (i2ccmd <> 0)

  return i2cresult                                              ' return ACK or NAK


pub read(ackbit) : b

'' Read byte from I2C bus

  rd_block(@i2cresult, 1, ackbit)

  return i2cresult & $FF


pub rd_byte(ackbit) : b

'' Read byte from I2C bus

  rd_block(@i2cresult, 1, ackbit)

  return i2cresult & $FF


pub rd_word(ackbit) : w

'' Read byte from I2C bus

  rd_block(@i2cresult, 2, ackbit)

  return i2cresult & $FFFF


pub rd_long(ackbit) : l

'' Read byte from I2C bus

  rd_block(@i2cresult, 4, ackbit)

  return i2cresult


pub rd_block(p_dest, count, ackbit) | cmd

'' Read block of count bytes from I2C bus to p_dest

  cmd.byte[0] := I2C_READ | (ackbit << 7)
  cmd.byte[1] := count
  cmd.word[1] := p_dest

  i2ccmd := cmd
  repeat while (i2ccmd <> 0)


pub stop

'' Create I2C stop sequence

  i2ccmd := I2C_STOP
  repeat while (i2ccmd <> 0)


dat { high-speed i2c }

                        org       0

fast_i2c                mov       outa, #0                      ' clear outputs
                        mov       dira, #0
                        rdlong    t1, par                       ' read pins and delaytix
                        mov       t2, t1                        ' copy for scl pin
                        and       t2, #$1F                      ' isolate scl
                        mov       sclmask, #1                   ' create mask
                        shl       sclmask, t2
                        mov       t2, t1                        ' copy for sda pin
                        shr       t2, #8                        ' isolate scl
                        and       t2, #$1F
                        mov       sdamask, #1                   ' create mask
                        shl       sdamask, t2
                        mov       delaytix, t1                  ' copy for delaytix
                        shr       delaytix, #16

                        mov       t1, #9                        ' reset device
:loop                   or        dira, sclmask
                        call      #hdelay
                        andn      dira, sclmask
                        call      #hdelay
                        test      sdamask, ina          wc      ' sample sda
        if_c            jmp       #cmd_exit                     ' if high, exit
                        djnz      t1, #:loop
                        jmp       #cmd_exit                     ' clear parameters

get_cmd                 rdlong    t1, par               wz      ' check for command
        if_z            jmp       #get_cmd

                        mov       tcmd, t1                      ' copy to save data
                        and       t1, #%111                     ' isolate command

                        cmp       t1, #I2C_START        wz
        if_e            jmp       #cmd_start

                        cmp       t1, #I2C_WRITE        wz
        if_e            jmp       #cmd_write

                        cmp       t1, #I2C_READ         wz
        if_e            jmp       #cmd_read

                        cmp       t1, #I2C_STOP         wz
        if_e            jmp       #cmd_stop

cmd_exit                mov       t1, #0                        ' clear old command
                        wrlong    t1, par
                        jmp       #get_cmd



cmd_start               andn      dira, sdamask                 ' float SDA (1)
                        andn      dira, sclmask                 ' float SCL (1, input)
                        nop
:loop                   test      sclmask, ina          wz      ' scl -> C
        if_z            jmp       #:loop                        ' wait while low
                        call      #hdelay
                        or        dira, sdamask                 ' SDA low
                        call      #hdelay
                        or        dira, sclmask                 ' SCL low
                        call      #hdelay
                        jmp       #cmd_exit



cmd_write               mov       tcount, tcmd                  ' (tcount := tcmd.byte[1])
                        shr       tcount, #8                    ' remove cmd byte
                        and       tcount, #$FF                  ' isolate count
                        mov       thubsrc, tcmd                 ' (thubsrc := tcmd.word[1])
                        shr       thubsrc, #16                  ' isolate p_src
                        mov       tackbit, #ACK                 ' assume okay

:byteloop               rdbyte    t2, thubsrc                   ' get byte
                        add       thubsrc, #1                   ' increment source pointer
                        shl       t2, #24                       ' position msb
                        mov       tbits, #8                     ' prep for 8 bits out

:bitloop                rcl       t2, #1                wc      ' bit31 -> carry
                        muxnc     dira, sdamask                 ' carry -> sda
                        call      #hdelay                       ' hold a quarter period
                        andn      dira, sclmask                 ' clock high
                        call      #hdelay
                        or        dira, sclmask                 ' clock low
                        djnz      tbits, #:bitloop

                        ' read a  ck/nak

                        andn      dira, sdamask                 ' make SDA input
                        call      #hdelay
                        andn      dira, sclmask                 ' SCL high
                        call      #hdelay
                        test      sdamask, ina          wc      ' test ackbit
        if_c            mov       tackbit, #NAK                 ' mark if NAK
                        or        dira, sclmask                 ' SCL low
                        djnz      tcount, #:byteloop

                        mov       thubdest, par
                        add       thubdest, #12                 ' point to i2cresult
                        wrlong    tackbit, thubdest             ' write ack/nak bit
                        jmp       #cmd_exit



cmd_read                mov       tackbit, tcmd                 ' (tackbit := tcmd.bit[7])
                        shr       tackbit, #7                   ' remove cmd
                        and       tackbit, #1                   ' isolate
                        mov       tcount, tcmd                  ' (tcount := tcmd.byte[1])
                        shr       tcount, #8                    ' remove ack/nak + cmd
                        and       tcount, #$FF                  ' isolate count
                        mov       thubdest, tcmd                ' (thubdest := tcmd.word[1])
                        shr       thubdest, #16                 ' isolate

:byteloop               andn      dira, sdamask                 ' make SDA input
                        mov       t2, #0                        ' clear result
                        mov       tbits, #8                     ' prep for 8 bits

:bitloop                call      #qdelay
                        andn      dira, sclmask                 ' SCL high
                        call      #hdelay
                        shl       t2, #1                        ' prep for new bit
                        test      sdamask, ina          wc      ' sample SDA
                        muxc      t2, #1                        ' new bit to t2.bit0
                        or        dira, sclmask                 ' SCL low
                        call      #qdelay
                        djnz      tbits, #:bitloop

                        ' write   ack/nak

                        cmp       tcount, #1            wz      ' last byte?
        if_nz           jmp       #:ack                         ' if no, do ACK
                        xor       tackbit, #1           wz      ' test user test ackbit
:ack    if_nz           or        dira, sdamask                 ' ACK (SDA low)
:nak    if_z            andn      dira, sdamask                 ' NAK (SDA high)
                        call      #qdelay
                        andn      dira, sclmask                 ' SCL high
                        call      #hdelay
                        or        dira, sclmask                 ' SCL low
                        call      #qdelay

                        wrbyte    t2, thubdest                  ' write result to p_dest
                        add       thubdest, #1                  ' increment p_dest pointer
                        djnz      tcount, #:byteloop
                        jmp       #cmd_exit



cmd_stop                or        dira, sdamask                 ' SDA low
                        call      #hdelay
                        andn      dira, sclmask                 ' float SCL
                        call      #hdelay
:loop                   test      sclmask, ina          wz      ' check SCL for "stretch"
        if_z            jmp       #:loop                        ' wait while low
                        andn      dira, sdamask                 ' float SDA
                        call      #hdelay
                        jmp       #cmd_exit



hdelay                  mov       t1, delaytix                  ' delay half period
                        shr       t1, #1
                        add       t1, cnt
                        waitcnt   t1, #0
hdelay_ret              ret



qdelay                  mov       t1, delaytix                  ' delay quarter period
                        shr       t1, #2
                        add       t1, cnt
                        waitcnt   t1, #0
qdelay_ret              ret

' --------------------------------------------------------------------------------------------------

sclmask                 res     1                               ' pin masks
sdamask                 res     1
delaytix                res     1                               ' ticks in 1/2 cycle

t1                      res     1                               ' temp vars
t2                      res     1
tcmd                    res     1                               ' command
tcount                  res     1                               ' bytes to read/write
thubsrc                 res     1                               ' hub address for write
thubdest                res     1                               ' hub address for read
tackbit                 res     1
tbits                   res     1

                        fit     496


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


  Note from JonnyMac:

  If you intend to modify and re-publish, kindly take my name off of this code so that I
  don't get credit for your good work, nor get blamed for your mistakes.

}}