'' =================================================================================================
''
''   File....... jm_tm1637_demo.spin
''   Purpose.... Demonstration code for TM1637 modules
''   Author..... Jon "JonnyMac" McPhalen
''               Copyright (c) 2019-2025 Jon McPhalen
''               -- see below for terms of use
''   E-mail..... jon.mcphalen@gmail.com
''   Started....
''   Updated.... 05 DEC 2025
''
'' =================================================================================================


con { timing }

  _xinfreq = 5_000_000                                          ' use 5MHz crystal
  _clkmode = xtal1 + pll16x                                     ' x16 = 80MHz

  CLK_FREQ = (_clkmode >> 6) * _xinfreq                         ' system freq as a constant
  MS_001   = CLK_FREQ / 1_000                                   ' ticks in 1ms
  US_001   = CLK_FREQ / 1_000_000                               ' ticks in 1us


con { terminal }

  BR_TERM  = 115_200                                            ' terminal baud rate


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { app io pins }

  DIO      =  1  { I/O }                                        ' TM1637 display
  CLK      =  0  { O   }


con

  #true,  ON, OFF
  #false, NO, YES

' settings for 6-digit module with decimal points and buttons
' * tested with: https://www.amazon.com/gp/product/B06WWN9Z6J
{
  WIDTH    = 6                                                  ' display width
  DPOINTS  = YES                                                ' display with decimal points
  HAS_BTNS = YES                                                ' display has buttons
  COLON    = NO                                                 ' display with colon (width usually 4)
}

' settings for 4-digit module with center colon

  WIDTH    = 4
  DPOINTS  = NO
  HAS_BTNS = NO
  COLON    = YES


  DEBUG    = NO                                                ' allow serial debugging


obj

' main                                                          ' * master Spin cog
  time : "jm_time_80"                                           '   timing and delays (80MHz)
  prng : "jm_prng"                                              '   pseudo-random number generator
  fmt  : "jm_format"                                            '   numeric formatting
  io   : "jm_io"                                                '   essential io
  disp : "jm_tm1637"                                            '   tm1637 driver
  term : "jm_fullduplexserial"                                  ' * serial IO for terminal

' * uses cog when loaded


dat { pre-initialized }

  Message       byte    " "[WIDTH], "HELLO", " "[WIDTH], 0

  Animate       byte    %01011100[WIDTH], %01100011, %01011100[WIDTH]


var

  long  tm1637


pub main | col

  setup

  if (DEBUG == YES)
    wait_for_terminal(true, 250)
    term.str(string("TM1637 Demo", 13, "--> "))
    if (tm1637 == true)
      term.str(string("device found", 13, 13))
    else
      term.fstr1(string("error code: $%.2x\r\r"), tm1637)
      repeat

  repeat
    scroll_text(@Message)
    buttons_demo
    decimal_demo
    clock_demo
    fp_decimal_demo
    hexadecimal_demo
    octal_demo
    quarternary_demo
    binary_demo
    buffer_demo


pub scroll_text(p_str) | idx

  repeat idx from 0 to strsize(p_str)-WIDTH
    disp.str(0, p_str+idx, WIDTH)
    time.pause(125)


pub buttons_demo | t, btn

  if (HAS_BTNS <> YES)
    return

  disp.clear

  if (WIDTH == 6)
    disp.str(0, string("btn"), 3)

  t := cnt
  repeat 100
    btn := disp.read_btn
    disp.str(WIDTH-2, fmt.rjdec(btn, 2, " "), 2)
    waitcnt(t += constant(50 * MS_001))

  disp.clear


pub decimal_demo : value | idx

'' Show right-justified decimal

  disp.clear

  repeat idx from 1 to WIDTH
    value := (value * 10) + idx
    disp.str(0, fmt.rjdec(value, WIDTH, " "), WIDTH)
    time.pause(250)

  time.pause(1000)
  disp.clear


pub clock_demo | h, m, t

  if ((WIDTH <> 4) or (COLON <> YES))
    return

  disp.clear

  h := 12
  m := 35

' runs at 60x speed (1s = 1m)
' note: colon controlled with decimal point bit (7)

  t := cnt
  repeat 11
    disp.str(0, fmt.fpdec(h*100+m, 5, 2), 4)                    ' convert & display as decimal
    waitcnt(t += CLK_FREQ)
    if (++m == 60)                                              ' bump minutes w/ roll-over
      m := 0
      if (++h == 13)                                            ' bump hours w/ roll-over
        h := 1

  disp.clear


pub fp_decimal_demo | countdown, t

  if (DPOINTS <> YES)
    return

  countdown := 10_0

  disp.clear
  disp.str(WIDTH-3, fmt.fpdec(countdown, 4, 1), 3)

  time.pause(1000)

  t := cnt
  repeat
    disp.str(WIDTH-3, fmt.fpdec(countdown, 4, 1), 3)
    waitcnt(t += constant(100 * MS_001))
    if (--countdown < 0)
      quit

  time.pause(1000)
  disp.clear


pub hexadecimal_demo | p_str, idx, c

  disp.clear
  disp.set_char(0, "h", false)

  repeat 10
    p_str := fmt.fhex(prng.randomize($000, $FFF), 3)            ' randomize value
    repeat idx from 0 to 2                                      ' fix B and D if present
      c := byte[p_str+idx]
      if ((c == "B") or (c == "D"))
        byte[p_str+idx] += 32
    disp.str(WIDTH-3, p_str, 3)
    time.pause(500)

  disp.clear


pub octal_demo | p_str, idx, c

  disp.clear
  disp.set_char(0, "o", false)

  repeat 10
    p_str := fmt.foct(prng.randomize(%000_000_000, %111_111_111), 3)    ' randomize value
    disp.str(WIDTH-3, p_str, 3)
    time.pause(500)

  disp.clear


pub quarternary_demo | value, p_str

  disp.clear
  disp.set_char(0, "q", false)

  repeat value from %%000 to %%333
    p_str := fmt.fquart(value, 3)
    disp.str(WIDTH-3, p_str, 3)
    time.pause(200)

  disp.clear


pub binary_demo | value

'' Show 4-digit random hex on right side of display

  disp.clear
  disp.set_char(0, "b", false)

  repeat value from 0 to 7
    disp.str(WIDTH-3, fmt.fbin(value, 3), 3)
    time.pause(500)

  disp.clear


pub buffer_demo | idx

'' Write pre-built segments buffer to display

  repeat 2
    repeat idx from 0 to WIDTH+1
      disp.wr_buf(0, @Animate+idx, 6)
      time.pause(125)

  disp.clear


pub setup

'' Setup IO and objects for application

  time.start                                                    ' setup timing & delays

  prng.seed(-cnt, -cnt ~> 2, $CAFE_BABE, cnt <- 2, cnt)         ' seed randomizer

  io.start(0, 0)                                                ' clear all pins (master cog)

  tm1637 := disp.setup(CLK, DIO, WIDTH, 0)                      ' connect to display
  disp.clear
  disp.display_on(4)

  if (DEBUG == YES)
    term.tstart(BR_TERM)                                        ' start serial for terminal *


pub wait_for_terminal(clear, delay)

'' Wait for terminal to be open and key pressed

  term.rxflush
  term.rx
  if (clear)
    term.tx(term#CLS)
    time.pause(delay #> 0)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}