'' =================================================================================================
''
''   File....... jm_tm1637.spin
''   Purpose.... TM1637 display driver
''   Author..... Jon "JonnyMac" McPhalen
''               Copyright (c) 2019-2025 Jon McPhalen
''               -- see below for terms of use
''   E-mail..... jon.mcphalen@gmail.com
''   Started....
''   Updated.... 04 DEC 2025
''               -- added bit_delay() for slow displays
''                  * also works with FlexProp compiler
''
'' =================================================================================================

con

  VERSION  = 1_1_0


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { tm1637  }

  WR_DATA = $40                                                 ' write data
  WR_AUTO = $40                                                 ' auto increment
  RD_BTN  = $42                                                 ' read button
  WR_FIXD = $44                                                 ' fixed address

  DSP_OFF = $80                                                 ' display off
  DSP_ON  = $88                                                 ' display on

  WR_SEGS = $C0                                                 ' segments address base


obj

  smap : "jm_tm163x_segs7"                                      ' 7-segment maps


var

  byte  clk                                                     ' clock pin
  byte  dio                                                     ' data pin
  byte  size                                                    ' usually 4 or 6

  byte  buffer[6]                                               ' segments buffer


pub null

'' This is not an application


pub setup(cpin, dpin, ncols, brightness) : result

'' Connect to TM1637
'' -- returns true if device ACKs
'' -- cpin and dpin are clock and data pins
'' -- ncols is the number of columsn used (1..6)
'' -- brightness sets display brightness level (0..8)

  clk := cpin                                                   ' copy pins
  dio := dpin

  size := 2 #> ncols <# 6                                       ' set display size

  dira[dio] := 0                                                ' float both
  outa[dio] := 0                                                ' clear output bit
  dira[clk] := 0
  outa[clk] := 0

  result := (send_command(0) == 0)                              ' look for device

  if (result)
    display_on(brightness)


pub clear

'' Clear all digits in display

  fill($00)


pub fill(segments)

'' Fill display digits with segments

  bytefill(@buffer, segments, 6)

  send_command(WR_DATA)                                         ' auto increment address

  start
  write(WR_SEGS)                                                ' start at base
  repeat 6                                                      ' write all
    write(segments)
  stop


pub display_on(level)

'' Display on at brightness level
'' -- level is 0 (off), 1..8 to set brightness

  if (level =< 0)
    send_command(DSP_OFF)
  else
    send_command(DSP_ON | (1 #> level <# 8)-1)


pub set_segs(col, segs)

'' Write segments to specified column
'' -- col is column (0..5) to write segments

  if ((col < 0) or (col > 5))                                   ' validate column
    return

  buffer[col] := segs                                           ' save for dpoint maniputation

  start
  write(WR_SEGS | col)                                          ' set column
  write(segs)                                                   ' write the new segments
  stop


pub set_digit(col, value, dp)

'' Write digit to specified column
'' -- col is column (0..size-1)
'' -- value is 0..15
'' -- dp is decimal point control

  if ((col < 0) or (col > size))
    return

  value := smap.digit(value, 16)

  if (dp)
    value |= smap#SEG_DP

  set_segs(col, value)


pub set_char(col, ch, dp)

'' Write ASCII character to specified column
'' -- col is column (0..size-1)
'' -- ch is 32 (space) to 127
'' -- dp is decimal point control

  if ((col < 0) or (col > size))
    return

  ch := smap.ascii(ch)

  if (dp)
    ch |= smap#SEG_DP

  set_segs(col, ch)


con { special note on strings }

{
  If a decimal point follows another character it will modify that character, hence
  "Hi." will only occupy two characters in the display.
}


pub str(col, p_str, len) | ch, la, segs

'' Write string to display
'' -- col is starting column (0..size-1)
'' -- p_str is pointer to string
'' -- len is # of characters to print (0 for auto-size)
''    * decimal points do not count in length
''    * e.g., "1.2.3.4.5.6." is considered len = 6

  if ((col < 0) or (col => size))
    return

  if (len =< 0)
    len := strsize(p_str)                                       ' auto size

  send_command(WR_AUTO)                                         ' auto increment characters

  start

  write(WR_SEGS | col)                                          ' set first column

  repeat while (len)
    ch := byte[p_str++]                                         ' get a character
    if (ch == 0)                                                ' if at end
      quit                                                      '  we're done
    else                                                        ' else
      segs := smap.ascii(ch)                                    '  convert to segments
      if (ch <> ".")                                            '  if not leading .
        la := byte[p_str]                                       '   look ahead to next character
        if (la == ".")                                          '   if .
          segs |= smap#SEG_DP                                   '    add to segments
          ++p_str                                               '    move past .
      write(segs)                                               '  write the character
      if (++col == size)                                        '  update colum
        quit
      len -= 1                                                  '  update length

  stop


pub set_dpoint(col, state)

'' Set or clear the decimal point from element idx
'' -- col is column (0..size-1)

  if ((col < 0) or (col => size))
    return

  if (state)
    buffer[col] |= smap#SEG_DP
  else
    buffer[col] &= !smap#SEG_DP

  set_segs(col, buffer[col])


pub wr_buf(col, p_buf, len)

'' Copy segments buffer to display
'' -- col is starting column (0..5)
'' -- p_buf is pointer to segments buffer
'' -- len is # of disply elements to write

  if ((col < 0) or (col => size))
    return

  if (len < 1)
    return

  len <#= size-col                                              ' prevent overwrite

  send_command(WR_DATA)                                         ' auto increment

  start
  write(WR_SEGS | col)                                          ' set starting column
  repeat len
    buffer[col] := byte[p_buf++]
    write(buffer[col++])
  stop


pub get_segs(col) : segs

'' Return segments for specified column

  if ((col < 0) or (col => size))
    return 0
  else
    return buffer[col]


pub read_btn : btn

'' Returns btn value if pressed
'' -- 0 when no btn pressed
'' -- 1..16 is pressed key #
'' -- device does not allow multiple keys

  start
  write(RD_BTN)                                                 ' send read commnad
  btn := read                                                   ' check for pressed button
  stop

  if (btn == $FF)
    return 0                                                    ' no press
  else
    return (!btn & $1F) - 7                                     ' decode button #


pri send_command(cmd) : ackbit

  start
  ackbit := write(cmd)
  stop


pri start

  dira[dio] := 0                                                ' float both
  dira[clk] := 0
  bit_delay

  dira[dio] := 1                                                ' dio low
  bit_delay

  dira[clk] := 1                                                ' clock low
  bit_delay


pri write(b) : ackbit

  b := !b >< 32                                                 ' invert (for dira) & flip (lsb first)

  repeat 8
    dira[dio] := b <-= 1                                        ' put bit on dio (lsb first)
    dira[clk] := 0                                              ' clock high
    bit_delay
    dira[clk] := 1                                              ' clock low
    bit_delay

  dira[dio] := 0                                                ' make dio input
  dira[clk] := 0                                                ' clock high
  bit_delay
  ackbit := ina[dio]                                            ' grab ack bit (should be 0)
  dira[clk] := 1                                                ' clock low
  bit_delay


pri read : b

  dira[dio] := 0                                                ' input

  repeat 8
    dira[clk] := 0                                              ' clock high
    bit_delay
    b := (b << 1) | ina[dio]                                    ' get a bit
    dira[clk] := 1                                              ' clock low
    bit_delay

  dira[dio] := 1                                                ' output ack bit (0)
  dira[clk] := 0                                                ' clock high
  bit_delay
  dira[clk] := 1                                                ' clock low
  bit_delay

  b ><= 8                                                       ' correct for lsb first


pri stop

  dira[clk] := 1                                                ' both low
  dira[dio] := 1
  bit_delay

  dira[clk] := 0                                                ' clock high
  bit_delay

  dira[dio] := 0                                                ' dio high
  bit_delay


pri bit_delay

'' Half bit delay

  waitcnt(cnt + (clkfreq / 10_000))                             ' ~100us


con { license }

{{

  MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

}}