'' =================================================================================================
''
''   File....... jm_format.spin
''   Purpose.... Formatting routines for serialized output
''   Authors.... Chip Gracey, Jon McPhalen
''               -- Copyright (c) 2006-2025 Parallax, Inc., Jon McPhalen
''   E-mail.....
''   Started....    APR 2006
''   Updated.... 18 FEB 2020
''
'' =================================================================================================


con

  MAX_WIDTH = 32                                                ' maximum size of any string
  BUF_SIZE  = MAX_WIDTH+1                                       ' width chars + zero terminator


var

  byte  bidx                                                    ' index into buffer
  byte  sbuf[BUF_SIZE]                                          ' string buffer for numeric data


pub null

'' This is not a top-level object


pub dec(value)

'' Output value as left-justified decimal

  bidx := 0                                                     ' reset buffer index
  dec_str(value)                                                ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub rjdec(value, width, pchar)

'' Output value as right-justified decimal
'' -- fwidth is width of (padded) field for value
''    * must add 1 if field could be negative
'' -- pchar is [leading] pad character (usually "0" or " ")

  bidx := 0                                                     ' reset buffer index
  rjdec_str(value, width, pchar)                                ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub fpdec(value, fwidth, dp) | divisor, sign, whole

'' Output fixed-point decimal value
'' -- fwidth is width of entire field
''    * Use 0 for left-justification
''    * Warning: insufficient width can hang program
'' -- dp is the number of digits after the decimal point
''    * e.g., dp of 3 will result in a format of x.xxx

  if (fwidth < dp+2)                                            ' force left if narrow field
    fwidth := 0

  divisor := 1                                                  ' create advisor
  repeat dp
    divisor *= 10

  if (value => 0)
    sign := 0
  else
    sign := -1

  whole := value / divisor

  bidx := 0                                                     ' reset buffer index
  if (fwidth)
    rjdec_str(value / divisor, fwidth-(dp+1), " ")              ' whole part (right justified)
  else
    if (whole == 0) and (sign == -1)
      sbuf[bidx++] := "-"
    dec_str(value / divisor)                                    ' whole part (left justified)

  sbuf[bidx++] := "."                                           ' deciminal point

  rjdec_str(||value // divisor, dp, "0")                        ' fractional part
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub hex(value)

'' Output value as hexadecimal
'' -- width varies with value (1 to 8 digits)

  bidx := 0                                                     ' reset buffer index
  hex_str(value)                                                ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub fhex(value, digits)

'' Output value as hexadecimal
'' -- digits is 1 to 8

  bidx := 0                                                     ' reset buffer index
  fhex_str(value, digits)                                       ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub oct(value)

'' Output value as hexadecimal
'' -- width varies with value (1 to 11 digits)

  bidx := 0                                                     ' reset buffer index
  oct_str(value)                                                ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub foct(value, digits)

'' Output value as hexadecimal
'' -- digits is 1 to 11

  bidx := 0                                                     ' reset buffer index
  foct_str(value, digits)                                       ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub quart(value)

'' Output value as quarternary
'' -- width varies with value (1 to 16 digits)

  bidx := 0                                                     ' reset buffer index
  quart_str(value)                                              ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub fquart(value, digits)

'' Output value as quarternary
'' -- digits is 1 to 16

  bidx := 0                                                     ' reset buffer index
  fquart_str(value, digits)                                     ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub bin(value)

'' Output value as binary
'' -- width varies with value (1 to 32 digits)

  bidx := 0                                                     ' reset buffer index
  bin_str(value)                                                ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


pub fbin(value, digits)

'' Output value as binary
'' -- digits is 1 to 32

  bidx := 0                                                     ' reset buffer index
  fbin_str(value, digits)                                       ' convert
  sbuf[bidx] := 0                                               ' terminate buffer

  return @sbuf


con

' These are the formatting routines; the value passed will be converted to the
' format called at the current place in the buffer (indicated by bidx)
'
' Programming is responsible for width specifications -- do not exceed MAX_WIDTH


pri dec_str(value) | x, divisor

  x := (value == negx)                                          ' check for max negative
  if (value < 0)
    value := ||(value+x)                                        ' if negative, make positive; adjust for max negative
    sbuf[bidx++] := "-"                                         ' and output sign

  divisor := 1_000_000_000                                      ' initialize divisor

  repeat 10                                                     ' loop for 10 digits
    if (value => divisor)
      sbuf[bidx++] := value/divisor + "0" + x*(divisor == 1)    ' if non-zero digit, output digit; adjust for max negative
      value //= divisor                                         ' and digit from value
      result := true                                            ' flag non-zero found
    elseif (result or (divisor == 1))
      sbuf[bidx++] := "0"                                       ' if zero digit (or only digit) output it
    divisor /= 10                                               ' update divisor


pri rjdec_str(value, fwidth, pchar) | tval, pad

'  Original code by Dave Hein
'  Modifications by Jon McPhalen

  if (value => 0)                                               ' if positive
    tval := value                                               '  copy value
    pad := fwidth - 1 <# MAX_WIDTH                              '  make room for 1 digit
  else
    if (value == negx)                                          '  if max negative
      tval := posx                                              '    use max positive for width
    else                                                        '  else
      tval := -value                                            '    make positive
    pad := fwidth - 2                                           '  make room for sign and 1 digit

  repeat while (tval => 10)                                     ' adjust pad for value width > 1
    pad--
    tval /= 10

  repeat pad                                                    ' print pad
    sbuf[bidx++] := pchar

  dec_str(value)                                                ' convert value


pri hex_str(value) | digits

'' Add hex value to buffer

  if (value == 0)
    sbuf[bidx++] := "0"
  else
    digits := ((>|value-1) >> 2) + 1                            ' find high nib, calculate digits needed
    fhex_str(value, digits)


pri fhex_str(value, digits)

'' Add fixed-width hex value to buffer

  digits <#= 8

  value <<= (8 - digits) << 2
  repeat digits
    sbuf[bidx++] := lookupz((value <-= 4) & $F : "0".."9", "A".."F")


pri oct_str(value) | digits

'' Add octal value to buffer

  if (value == 0)
    sbuf[bidx++] := "0"
  else
    digits := ((>|value-1) / 3) + 1                             ' calculate digits needed
    foct_str(value, digits)


pri foct_str(value, digits)

'' Add fixed-width octal value to buffer

  if (digits => 11)
    sbuf[bidx++] := "0" + ((value <-= 2) & %011)                ' only 2 bits in high digit
    digits := 10
  else
    value <<= 32 - (digits * 3)

  repeat digits
    sbuf[bidx++] := "0" + ((value <-= 3) & %111)


pri quart_str(value) | digits

'' Add quarternary value to buffer

  if (value == 0)
    sbuf[bidx++] := "0"
  else
    digits := ((>|value-1) >> 1) + 1
    fquart_str(value, digits)


pri fquart_str(value, digits)

'' Add fixed-width quarternary value to buffer

  digits <#= 16

  value <<= (16 - digits) << 1
  repeat digits
    sbuf[bidx++] := "0" + ((value <-= 2) & %11)


pri bin_str(value)

'' Add binary value to buffer

  if (value == 0)
    sbuf[bidx++] := "0"
  else
    fbin_str(value, >|value)                                    ' digits is highest 1 bit position + 1


pri fbin_str(value, digits)

'' Add fixed-width binary value to buffer

  digits <#= 32

  value <<= 32 - digits
  repeat digits
    sbuf[bidx++] := "0" + ((value <-= 1) & %1)


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}