'' =================================================================================================
''
''   File........ jm_pasm-cog_from_file.spin
''   Purpose.....
''   Cogs Used...
''   Author...... Jon "JonnyMac" McPhalen
''                Copyright (c) 2026 Jon McPhalen
''                -- see below for terms of use
''   E-mail...... jon.mcphalen@gmail.com
''   Started.....
''   Updated..... 27 JAN 2026
''
'' =================================================================================================


con

  VERSION = 0_0_0


con { timing }

  _xinfreq = 5_000_000                                          ' use 5MHz crystal
  _clkmode = xtal1 + pll16x                                     ' x16 = 80MHz

  CLK_FREQ = (_clkmode >> 6) * _xinfreq                         ' system freq as a constant
  MS_001   = CLK_FREQ / 1_000                                   ' ticks in 1ms
  US_001   = CLK_FREQ / 1_000_000                               ' ticks in 1us


con { terminal }

  BR_TERM  = 115_200                                            ' terminal baud rate


con { fixed io pins }

  PGM_RX   = 31  { I }                                          ' serial / programming
  PGM_TX   = 30  { O }

  EE_SDA   = 29  { I/O }                                        ' i2c / eeprom
  EE_SCL   = 28  { I/O }

  LED_27   = 27  { O }                                          ' LEDs on PAB and FLiP
  LED_26   = 26  { O }


con { app io pins }

{
' JM adapter
' -------------
  SD_PWR   = 22  { L/X }
  SD_CS    = 20  { L/X }
  MOSI     = 19  { O }
  SCLK     = 18  { O }
  MISO     = 17  { I }
  SD_DET   = 16  { I }
}

' PAB
' -------------
  SD_PWR   = -1  { L/X }
  SD_CS    = 25  { L/X }
  MOSI     = 24  { O }
  SCLK     = 23  { O }
  MISO     = 22  { I }
  SD_DET   = -1  { I }


con

  #true,  ON, OFF
  #false, NO, YES

  RAM_BUF = $7800                                               ' upper 2K of hub RAM


obj

' main                                                          ' * master Spin cog
  time : "jm_time_80"                                           '   timing and delays (80MHz)
  io   : "jm_io"                                                '   simple io
  sd   : "fsrw"                                                 ' * uSD file IO
  term : "jm_fullduplexserial"                                  ' * serial IO for terminal

' * uses cog when loaded


dat { pre-initialized }

  Menu                  byte    "[D] Directory      ", 13
                        byte    "[T] Test file size ", 13
                        byte    "[P] PASM to File   ", 13
                        byte    "[F] File to cog    ", 13
                        byte    13
                        byte    0

  TestFile              byte    "p1code.bin", 0


var { globals }


pub main | k

  setup
  wait_for_terminal(true, 250)

  repeat
    term.tx(term#CLS)
    term.str(@Menu)

    k := term.rx
    case k
      "d", "D" : show_directory
      "t", "T" : show_test_file
      "p", "P" : pasm_to_file
      "f", "F" : file_to_cog

  repeat
    waitcnt(0)


pub show_directory

  term.tx(term#CLS)

  if (mounted)
    directory
  else
    term.fstr0(string("Error: uSD did not mount.\r"))

  press_a_key


pub show_test_file

  term.tx(term#CLS)
  term.fstr1(string("Test file: %s\r"), @TestFile)

  if (has_file(@TestFile))
    term.fstr1(string("-- Size: %d bytes\r"), sd.get_filesize)
  else
    term.fstr0(string("-- not found on SD\r"))

  close_file

  press_a_key


pub pasm_to_file | len, p_code, buf

  term.tx(term#CLS)
  term.fstr0(string("Writing PASM code to file:\r"))
  term.fstr1(string("-- %s\r"), @TestFile)

  len := (@end_of_pasm - @entry) >> 2                           ' # longs in pasm code

  delete_file(@TestFile)                                        ' delete old file

  open_file(@TestFile, "w")                                     ' open empty file

  p_code := @entry                                              ' point to code

  repeat len
    buf := long[p_code]                                         ' read instruction/data
    p_code += 4                                                 ' point to next
    wr_buf(@buf, 4)                                             ' write to file

  close_file

  open_file(@TestFile, "r")                                     ' re-open new file
  term.fstr1(string("-- %d bytes\r"), sd.get_filesize)          ' show bytes written
  close_file

  press_a_key


pub file_to_cog | fsize, p_code, buf, flashpin, flashtix

'' Load code into upper RAM, then launch into cog

  term.tx(term#CLS)
  term.fstr0(string("Reading PASM code from file:\r"))
  term.fstr1(string("-- %s\r"), @TestFile)

  ifnot (has_file(@TestFile))
    term.fstr0(string("-- not found on SD\r"))
    press_a_key

  fsize := sd.get_filesize
  term.fstr1(string("-- %d bytes\r"), fsize)

  p_code := RAM_BUF

' warning: no checks!!!

  repeat (fsize >> 2)                                           ' read longs
    rd_buf(@buf, 4)
    long[p_code] := buf
    p_code += 4

  close_file

  flashpin := 27
  flashtix := CLK_FREQ >> 3

  cognew(RAM_BUF, @flashpin)

  press_a_key


pub press_a_key

  term.fstr0(string("\rPress any key..."))

  term.rxflush
  term.rx


pub setup : check

'' Setup IO and objects for application

  time.start                                                    ' setup timing & delays

  io.start(0, 0)                                                ' clear all pins (master cog)

  mount_sd

  term.tstart(BR_TERM)                                          ' start serial for terminal *


pub wait_for_terminal(clear, mshold)

'' Wait for terminal to be open and key pressed

  term.rxflush
  term.rx
  if (clear)
    term.tx(term#CLS)
    time.pause(mshold #> 0)


con { file io }

  SD_OK = 0


var

  long  mounted

  byte  filename[16]                                            ' 8.3 plus 0
  byte  linebuf[81]                                             ' line buffer


pub mount_sd : check

'' Attempt to mount SD card
'' -- returns true if SD card successfully mounted

  if (SD_PWR > -1)
    io.low(SD_PWR)                                              ' for JM uSD adapter
    time.pause(50)

  if (SD_DET > -1)                                              ' for JM usd adapter
    check := (ina[SD_DET] == 0)                                 ' card inserted?
  else
    check := true

  if (check)
    check   := \sd.mount_explicit(MISO, SCLK, MOSI, SD_CS)
    mounted := (check == SD_OK)

  return mounted


pub dismount_sd

'' Unmounts SD card

  \sd.unmount
  mounted := false


pub has_file(p_str) : check

'' Returns true if file is on SD card

  check := \sd.popen(p_str, "r")
  if (check == 0)
    return true
  else
    return false


pub open_file(p_str, mode) : check

'' Attempt to open file
'' -- p_str is pointer to filename (z-string)
'' -- mode is [lower-case] single character, (e.g., "r" for read)
'' -- sd card must be mounted before open

  if (mounted)
    check := \sd.popen(p_str, mode)                             ' attempt to open
    if (check == 0)
      if (mode == "r")
        \sd.seek(0)                                             ' force sector read
      return true
    else
      return false
  else
    return false


pub delete_file(p_str) : check

'' Delete file if on uSD
'' -- p_str is pointer to filename (z-string)
'' -- sd card must be mounted before open

  if (mounted)
    check := \sd.popen(p_str, "d")                              ' attempt to delete
    if (check == SD_OK)
      return true
    else
      return false
  else
    return false


pub close_file

'' Closes open file

  \sd.pclose


pub rd_line(p_str, n) | c, len

'' Reads line of text from open text file
'' -- terminated by LF or EOF
'' -- p_str is address of string buffer
'' -- n is maximum number of characters to read from line

  len := 0                                                      ' index into string

  repeat n
    c := \sd.pgetc                                              ' get a character
    if (c < 0)                                                  ' end of file
      if (len == 0)                                             ' if nothing
        return -1                                               ' return empty
      else
        quit

    if (c <> 10)                                                ' if not LF char
      byte[p_str++] := c                                        '  add to buffer
      len++
    else                                                        ' if LF
      quit                                                      '  we're done

  byte[p_str] := 0                                              ' terminate end of line

  return len


pub wr_line(p_str, usecr, uself)

'' Writes z-string to open text file
'' -- set usecr to true to add CR
'' -- set uself to true to add LF

  \sd.pputs(p_str)
  if (usecr)
    \sd.pputc(13)
  if (uself)
    \sd.pputc(10)


pub rd_buf(p_buf, n) | check

'' Reads n bytes from file to buffer at p_buf
'' -- returns number of bytes read

  return \sd.pread(p_buf, n)


pub wr_buf(p_buf, n) | check

'' Writes n bytes to file from buffer at p_buf
'' -- returns number of bytes written

  return \sd.pwrite(p_buf, n)


pub directory | fcount, check, idx, p_fname

'' Show directory (root only) of uSD card

  sd.opendir                                                    ' setup for .nextfile
  fcount := 0                                                   ' reset files count

  term.fstr0(string("Directory\r\r"))

  repeat
    bytefill(@filename, 0, 13)
    check := \sd.nextfile(@filename)                            ' get next file in root
    if (check == SD_OK)                                         ' if found
      fcount += 1                                               '  increment file count
      term.fstr1(string("  %s\r"), @filename)                   '  show name
    else                                                        ' else
      quit                                                      '  done

  if (fcount > 0)
    term.tx(13)
    term.dec(fcount)
    term.fstr0(string(" file(s) found.\r"))
  else
    term.fstr0(string("No files found.\r"))

  sd.pclose


con { test code }

dat { cog driver }

                        org       0

entry                   mov       t1, par                       ' point to parameters
                        rdlong    t2, t1                        ' read pin
                        mov       pinmask, #1                   ' convert to mask
                        shl       pinmask, t2
                        or        dira, pinmask                 ' set to output
                        add       t1, #4
                        rdlong    delayticks, t1                ' get 1/2 cycle ticks

                        mov       t1, cnt                       ' sync blink timer
                        add       t1, delayticks                ' set initial delay

loop                    or        outa, pinmask                 ' led pin high
                        waitcnt   t1, delayticks
                        andn      outa, pinmask                 ' led pin low
                        waitcnt   t1, delayticks
                        jmp       #loop
                                       '
' --------------------------------------------------------------------------------------------------

pinmask                 res       1
delayticks              res       1

t1                      res       1                             ' work vars
t2                      res       1
t3                      res       1

end_of_pasm


con { license }

{{

  Terms of Use: MIT License

  Permission is hereby granted, free of charge, to any person obtaining a copy of this
  software and associated documentation files (the "Software"), to deal in the Software
  without restriction, including without limitation the rights to use, copy, modify,
  merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be included in all copies
  or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
  INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
  PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
  OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

}}