{{
@file memory.spin
List Memory Manager
}}

{{
  List chunks are allocated from "heap" start down (decrementing list).
  Each chunk length is a power of 2. The minimum chunk size is MINCHUNKSIZE.
  The new entry is the heaphead so it is the first accessed.
  The address of the new entry is the *next* address of the next new entry.
  The data for the new entry is the address + 2 for word wide addresses.
  An entry is free if bit 0 of the next pointer is clear, else it is used.
  Entries are word aligned (value & -2 = value & $fffe or value & $ffffffe).
}}

  MINCHUNKSIZE = $08            'minimum heap allocation chunk size 8 is abs minimum
  HEADLENGTH = 2                'heap list header length in bytes
  
{{
con
  _clkmode = xtal1 + pll16x
  _xinfreq = 5_000_000

pub main | ptr, n, v
'' Initialize heap and test private alloc/free
  sx.start(31,30,0,115200)
  waitcnt(clkfreq*1+cnt)
  sx.out(0)
  
  init($7f04)

  malloc(7)
  ptr := malloc(7)
  free(ptr)
  malloc(7)
  malloc(7)
  
  repeat n from 0 to 511
    if ptr
      free(ptr)
    ptr := malloc((v?) & $7ff)
    ifnot ptr
      quit
  repeat
'}}  

dat
heaphead  long 0

obj
'  sx    : "FullDuplexSingleton"
  
pub init(headp) | start, end
  heaphead := headp-4
  heaphead &= -2 
{
  sx.str(string($d,$d,"Alloc Init $"))
  sx.hex(heaphead, 4)
  sx.out($d)
'}
  word[heaphead] := 0           'setNext(heaphead,0)

pub malloc(num) | dataptr, sp, stkptr
'' user alloc
'' allocates a block of num bytes memory in a list
'' saves block pointer to Java HeapPointer
''

{
  sx.str(string($d,"malloc "))
  sx.dec(num)
  sx.out($d)
'}

  dataptr := palloc(num)
  ifnot dataptr                 ' forceGC now done in PASM ... reenable here is a waste
{
    sx.str(string("Out of Memory! "))
    sx.hex(heaphead,4)
}
    repeat
    return 0

  return dataptr

pub free(ptr) | nxtptr
'' user free memory
'' returns zero on error
''
  nxtptr := heaphead
  repeat while word[nxtptr] & -2 'getNext(nxtptr)
    if nxtptr == ptr-2 'getLinkPtr(ptr)
      if getUsed(nxtptr)
        word[nxtptr] &= -2      ' setFree(nxtptr)
      return true
    nxtptr := word[nxtptr] & -2 'getNext(nxtptr)

  return false

pri getDataPtr(linkptr)
  return linkptr+2

pri getLinkPtr(dataptr)
  return dataptr-2

pri isFree(linkPtr)
  return !getUsed(linkPtr) & 1

pri getUsed(linkPtr)
  ifnot linkPtr
    return 0
  return word[linkPtr] & 1

pri getSize(linkPtr)
  if linkPtr
    return ((word[linkPtr] & -2) - linkPtr) '(getNext(linkPtr)-linkPtr)
  return 0

pri getDataSize(linkPtr)
  if linkPtr
    return ((word[linkPtr] & -2) - linkPtr - headlength) '(getNext(linkPtr)-linkPtr-headlength)
  return 0

pri chunksize(num) | tmp
  result := num
  num := >| num
  result := 1 << (num)
  if result < MINCHUNKSIZE
    result := MINCHUNKSIZE

pri additem(headptr, num) | newptr
'' add new pointer to list
  num := chunksize(num + headlength)
  newptr :=(headptr - num)      ' always enough space to make even
  newptr &= -2                  ' even address
  word[newptr] := headptr       ' setNext(newptr, headptr)      ' headptr->next end->next = new pointer
  word[newptr] |= 1             ' setUsed(newptr)               ' entry is used
  headptr := newptr             ' nextptr is now headptr
  return headptr 

pri insert(prvptr, num) | newptr

'' special case of add
'' insert new pointer to list

  num := chunksize(num + headlength)

  newptr :=(prvptr + num)       ' always enough space to make even
  newptr &= -2                  ' even address
  
  word[prvptr] |= 1             'setUsed(prvptr)               ' entry is used regardless of insert/replace

  if newptr <> word[prvptr] & -2 'getNext(prvptr)  ' if prev free block is split, keep leftovers free
    word[newptr] := word[prvptr] & -2 'getNext(prvptr)) ' newptr.next = prvptr.next
    word[newptr] &= -2           ' setFree(newptr)             ' new ptr is free
    word[newptr] := newptr
    
  return prvptr

pri palloc(numbytes) | datptr, nxtptr, prvptr, tmpptr, len

'' allocate memory using a linked list

  nxtptr := heaphead
  prvptr := 0

  repeat while word[nxtptr] & -2 'getNext(nxtptr)

    if prvptr
      if isFree(prvptr) 
        if (chunksize(numbytes+HEADLENGTH) == ((word[prvptr] & -2) - prvptr))     ' getSize(prvptr)
          prvptr := insert(prvptr, numbytes)
          return prvptr+2 'getDataPtr(prvptr)

    prvptr := nxtptr
    nxtptr := word[nxtptr] & -2 'getNext(nxtptr)

  'showFreeStats                 ' show only if garbage not recycled

  ' if no insert, additem to end of list
  heaphead := additem(heaphead, numbytes)
  return heaphead+2 'getDataPtr(heaphead)

{
pri showFreeStats | n
  sx.str(string($d," free stats: "))
  repeat n from 0 to BSTATLEN-1
    'sx.hex(n, 2) ' n 0 keeps $10 count, n1 keeps $20 count, n3 keeps $80 count ....
    'sx.out(":")
    sx.hex(bstats[n], 2)
    sx.out(" ")
'}

pub show | nextp, len, top
{
  sx.str(string($d,"Show Objects: "))
  sx.str(string("headptr "))
  sx.hex(heaphead,4)
  sx.out($d)
  len := 0
  nextp := heaphead             ' print used list
  repeat while word[nextp] & -2 ' getNext(nextp)   ' never print top of heap entry
  '{
    if(word[nextp] & -2 <> 0)
      sx.str(string(" Used $"))
    if(isFree(nextp) <> 0)
      sx.str(string(" Free $"))
    sx.hex(getDataPtr(nextp),4)
    sx.str(string(" Size $"))
    sx.hex(getSize(nextp),4)
    sx.str(string(" @ $"))
    sx.hex(nextp,4)
    sx.str(string(" -> $"))
    sx.hex(word[nextp] & -2 ,4)
    sx.out($d)
  '}
    nextp := word[nextp] & -2
    len++
  sx.str(string(" Object Count: "))
  sx.dec(len)
  sx.out($d)
'}

