/**
 * @file PLoadLib.c
 *
 * Downloads an image to Propeller using Windows32 API.
 * Ya, it's been done before, but some programs like Propellent and
 * PropTool do not recognize virtual serial ports. Hence, this program.
 *
 * Copyright (c) 2009 by John Steven Denson
 *
 * MIT License                                                           
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * 
 */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/timeb.h>
#include <unistd.h>
#include <termios.h>

#include "inttypes.h"
#include "ComPort.h"
#include "DebugLib.h"

#ifndef LINUX
static unsigned long getms()
{
    LARGE_INTEGER ticksPerSecond;
    LARGE_INTEGER tick;   // A point in time
    LARGE_INTEGER time;   // For converting tick into real time
    // get the high resolution counter's accuracy
    QueryPerformanceFrequency(&ticksPerSecond);
    if(ticksPerSecond.QuadPart < 1000) {
        printf("Your system does not meet timer requirement. Try another computer. Exiting program.\n");
        exit(1);
    }
    // what time is it?
    QueryPerformanceCounter(&tick);
    time.QuadPart = (tick.QuadPart*1000/ticksPerSecond.QuadPart);
    return (unsigned long)(time.QuadPart);
}

static void waitms(int ms)
{
    unsigned long t = getms();
    while((t+ms+10) > getms())
        ;
}

#endif

/**
 * sleep for ms milliseconds
 * @param ms - time to wait in milliseconds
 */
void msleep(int ms)
{
#ifndef LINUX
    waitms(ms);
#else
    volatile struct timeb t0, t1;
    do {
        ftime((struct timeb*)&t0);
        do {
            ftime((struct timeb*)&t1);
        } while (t1.millitm == t0.millitm);
    } while(ms-- > 0);
#endif
}

#ifdef LINUX
#define strupr strtoupper
#endif

char* strtoupper(char* s)
{
    int len = strlen(s);
    while(--len >= 0)
        s[len] = toupper(s[len]);
    return s;
}


static const int SHUTDOWN_CMD = 0;
static const int DOWNLOAD_RUN_BINARY = 1;
static const int DOWNLOAD_EEPROM = 2;
static const int DOWNLOAD_RUN_EEPROM = 3;

uint8_t LFSR = 80; // 'P'
int iterate(void)
{
    int bit = LFSR & 1;
    LFSR = (uint8_t)((LFSR << 1) | (LFSR >> 7 ^ LFSR >> 5 ^ LFSR >> 4 ^ LFSR >> 1) & 1);
    return bit;
}

/**
 * getBit ... get bit from serial stream
 * @param hSerial - file handle to serial port
 * @param status - pointer to transaction status 0 on error
 * @returns bit state 1 or 0
 */
int getBit(HANDLE hSerial, int* status)
{
    uint8_t mybuf[2];
    int rc = rx(hSerial, mybuf, 1);
    if(status)
        *status = rc;
    return *mybuf & 1;
}

/**
 * getAck ... get ack from serial stream
 * @param hSerial - file handle to serial port
 * @param status - pointer to transaction status 0 on error
 * @returns bit state 1 or 0
 */
int getAck(HANDLE hSerial, int* status)
{
    uint8_t mybuf[2];
    int rc = rx(hSerial, mybuf, 1);
    if(status)
        *status = rc;
    return *mybuf & 1;
}

/**
 * decodelong ... decode a string to a long word
 * @param buff - uint8_t buffer
 * @param len - length of string
 * @returns long word
 */
uint32_t decodelong(uint8_t* buff, int len)
{
    int n = 0;
    int b4,b2,b1;
    uint32_t data = 0;
    uint32_t end = 0;
    for(n = 0; n < len-1; n++) {
        b4 = ((buff[n] & 0x7d) & (4 << 4)) >> 4;
        b2 = ((buff[n] & 0x7d) & (2 << 2)) >> 2;
        b1 = ((buff[n] & 0x7d) & 1);
        data |= (b1 | b2 | b4) << (n*3);
        //printf("\n0x%08x: %02x %02x %02x", data, b4, b2, b1);
    }
    b2 = ((buff[len-1] & 0x7d) & (2 << 2)) >> 2;
    b1 = ((buff[len-1] & 0x7d) & 1) >> 0;
    end = (b1 | b2) << 30;
    data |= end;
    //printf("\n0x%08x.", data);
    return 0;
}

/**
 * makelong ... make an encoded long word to string
 * @param data - value to send
 * @param buff - uint8_t buffer
 * @returns nada
 */
void makelong(uint32_t data, uint8_t* buff)
{
    int n = 0;
    //printf("\n0x%08x: ", data);
    for( ; n < 10; n++) {
        buff[n] = (uint8_t)(0x92 | (data & 1) | ((data & 2) << 2) | ((data & 4) << 4));
        data >>= 3;
    }
    buff[n] = (0xf2 | (data & 1) | ((data & 2) << 2));
    //for(n = 0; n < 11; n++) printf("0x%02x ",buff[n]);
    //decodelong(buff,11);
}

/**
 * sendlong ... transmit an encoded long word to propeller
 * @param hSerial - file handle to serial port
 * @param data - value to send
 * @returns number of bytes sent
 */
int sendlong(HANDLE hSerial, uint32_t data)
{
    uint8_t mybuf[12];
    makelong(data, mybuf);
    return tx(hSerial, mybuf, 11);
}

/**
 * encodelong ... encode a long word to buffer for propeller
 * @param hSerial - file handle to serial port
 * @param data - value to send
 * @param bigbuf - buffer for encode ... user controls size
 * @param position - buffer position for encode
 * @returns next position for bigbuf
 */
int encodelong(HANDLE hSerial, uint32_t data, uint8_t* bigbuf, int position)
{
    makelong(data, bigbuf);
    return position+11;
}

/**
 * hwfind ... find propeller using sync-up sequence.
 * @param hSerial - file handle to serial port
 * @returns zero on failure
 */
int hwfind(HANDLE hSerial)
{
    int  n, ii, jj, rc, to;
    uint8_t mybuf[300];

    msleep(100); // pause after reset

    mybuf[0] = 0xF9;
    LFSR = 'P';  // P is for Propeller :)

    // set magic propeller byte stream
    for(n = 1; n < 251; n++)
        mybuf[n] = iterate() | 0xfe;
    tx(hSerial, mybuf, 251);

    // send gobs of 0xF9 for id sync-up
    for(n = 0; n < 258; n++)
        mybuf[n] = 0xF9;
    tx(hSerial, mybuf, 258);

    //for(n = 0; n < 250; n++) printf("%d ", iterate() & 1);
    //printf("\n\n");

    msleep(100);

    // wait for response so we know we have a Propeller
    for(n = 0; n < 250; n++) {
        to = 0;
        do {
            ii = getBit(hSerial, &rc);
        } while(rc == 0 && to++ < 100);
        //printf("%d ", rc);
        if(to > 100) {
            printf("Timeout waiting for response bit. Propeller Not Found!\n");
            return 0;
        }
        jj = iterate();
        //printf("%d:%d ", ii, jj);
        if(ii != jj) {
            printf("Lost HW contact. %d %x\n", n, *mybuf & 0xff);
            return 0;
        }
    }

    //printf("Propeller Version ... ");
    rc = 0;
    for(n = 0; n < 8; n++) {
        rc >>= 1;
        rc += getBit(hSerial,0) ? 0x80 : 0;
    }
    //printf("%d\n",rc);
    return rc;
}

/**
 * find a propeller on port
 * @param hSerial - file handle to serial port
 * @param sparm - pointer to DCB serial control struct
 * @param port - pointer to com port name
 * @returns non-zero on error
 */
int findprop(HANDLE hSerial, char* port)
{
    int version = 0;
    hwreset(hSerial);
    version = hwfind(hSerial);
    if(version) {
        printf("Propeller Version %d on %s\n", version, port);
    }
    return version != 0 ? 0 : 1;
}

/**
 * Upload file image to propeller on port.
 * A successful call to findprop must be completed before calling this function.
 * @param hSerial - file handle to serial port
 * @param dlbuf - pointer to file buffer
 * @param count - number of bytes in image
 * @param type - type of upload
 * @returns non-zero on error
 */
int upload(HANDLE hSerial, uint8_t* dlbuf, int count, int type)
{
    int  n  = 0;
    int  rc = 0;
    int  wv = 100;
    uint32_t data = 0;

    int  sendone = 1; // send one at a time ... Propeller can't handle a bigbuf blast.
    int  position = 0;
    int  longcount = count/4;

    uint8_t* bigbuf = (uint8_t*)malloc(count*11+1); // for encodelong -> tx bigbuf only.

    // send type
    if(sendlong(hSerial, type) == 0) {
        printf("sendlong type failed.\n");
        return 1;
    }


    // send count
    if(sendlong(hSerial, longcount) == 0) {
        printf("sendlong count failed.\n");
        return 1;
    }
    //if(1) {
    //} else position = encodelong(hSerial, longcount, bigbuf, position);

    printf("Writing %d bytes to %s.\n",count,(type == DOWNLOAD_RUN_BINARY) ? "Propeller RAM":"EEPROM");
    //msleep(100);

    for(n = 0; n < count; n+=4) {
        //printf("%d ",n);
        data = dlbuf[n] | (dlbuf[n+1] << 8) | (dlbuf[n+2] << 16) | (dlbuf[n+3] << 24) ;
        if(sendone) {
            if(sendlong(hSerial, data) == 0) {
                printf("sendlong error");
                free(bigbuf);
                return 1;
            }
        }
        else position = encodelong(hSerial, data, bigbuf, position);
        //if(n % 0x40 == 0) putchar('.');
    }
    if(!sendone) tx(hSerial, bigbuf, position);

    msleep(150);                // wait for checksum calculation on Propeller ... 95ms is minimum
    sendlong(hSerial, 0xF9);

    printf("Verifying ... ");
    fflush(stdout);

    for(n = 0; n < wv; n++) {
        msleep(25);
        getAck(hSerial,&rc);
        if(rc) break;
    }
    if(n >= wv) printf("Upload Timeout Error!\n");
    else        printf("Upload OK!\n");

    free(bigbuf);
    return 0;
}

/**
 * find and load propeller with file - assumes port is already open
 * @returns non-zero on error.
 */
int pload(char* file, char* port)
{
    int rc = 0;
    int count;
    int type = DOWNLOAD_RUN_BINARY;
    uint8_t dlbuf[0x8000]; // 32K limit
    HANDLE hSerial;

    // read program if file parameter is available
    //
    FILE* fp = fopen(file, "rb");
    if(!fp) {
        printf("Can't open '%s' for upload.\n", file);
        return 4;
    }
    else {
        count = fread(dlbuf, 1, 0x8000, fp);
        printf("Downloading %d bytes of '%s'\n", count, file);
        fclose(fp);
    }
    if(strstr(strtoupper(file), ".EEPROM") > 0) type = DOWNLOAD_RUN_EEPROM;

    hSerial = gser;
    //hSerial = openport(port);
    //printf("Port %s %d\n", port, hSerial);

    //
    // find propeller
    //
    rc = findprop(hSerial, port);

    // if found and file, upload file
    //
    if(!rc && file) {
        rc = upload(hSerial, dlbuf, count, type);
    }

    //closeport(hSerial);

    return rc;
}

HANDLE gser;

#ifndef INVALID_HANDLE_VALUE
#define INVALID_HANDLE_VALUE 0
#endif

char* appe_str  = "A\n";
char* dump_str  = "D\n";
char* erasestr  = "E\n";
char* save_str  = "S\n";
char* countstr  = "C\n";
char* resetstr  = "R\n";
char* OKstr     = "OK>";

#define MAXRBUF 80

static void flush(HANDLE hSerial)
{
    int rc;
    uint8_t rbuf[MAXRBUF+1];

    do {
        rc = rx(hSerial,rbuf,MAXRBUF);
        if(rc < 0) {
            printf("Unexpected port error!\n");
            return;
        }
        if(0 && rc > 0) {
            rbuf[rc] = '\0';
            printf("Flush received: '%s'\n",rbuf);
        }
    } while(rc > 0);
}

static void waitus(HANDLE hSerial, int us)
{
    struct timeval tv;
    fd_set fd;
    FD_ZERO(&fd);
    FD_SET(hSerial,&fd);
    tv.tv_sec = 0;
    tv.tv_usec = us;
    select(1,&fd,0,0,&tv);
}

static int expect(HANDLE hSerial, char* cmd, char* resp)
{
    int  rc;
    int  n = 0;
    char rbuf[MAXRBUF+1];
    char buf[10];   // read 1 char at a time

    if(strlen(cmd) > 0) {
        //printf("\nCommand %s", cmd);
        write(hSerial, cmd, strlen(cmd));
    }

    if(strlen(resp) > 0) {
        rbuf[n] = '\0';
        while(1) {
            msleep(10);
            //waitus(hSerial,5000);
            rc = read(hSerial, buf, 1);
            if(rc < 0) {
                printf("Unexpected port error!\n");
                return rc;
            }
            else if(rc > 0) {
                rbuf[n++] = *buf;
                rbuf[n] = '\0';
                if(strstr((char*)rbuf,resp)) {
                    //printf("%s",rbuf);
                    break;
                }
                if(strstr((char*)rbuf,"ERROR")) {
                    printf("Expect ERROR: '%s'",rbuf);
                    return -1;
                }
            }
        }
    }
    return rc;
}

static int expecttime(HANDLE hSerial, char* cmd, char* resp, int time)
{
    int  rc;
    int  n = 0;
    char rbuf[MAXRBUF+1];
    char buf[10];   // read 1 char at a time

    if(strlen(cmd) > 0) {
        //printf("\nCommand %s", cmd);
        write(hSerial, cmd, strlen(cmd));
    }

    if(strlen(resp) > 0) {
        rbuf[n] = '\0';
        while(--time) {
            msleep(10);
            rc = read(hSerial, buf, 1);
            if(rc < 0) {
                printf("Unexpected port error!\n");
                return rc;
            }
            else if(rc > 0) {
                rbuf[n++] = *buf;
                rbuf[n] = '\0';
                if(strstr((char*)rbuf,resp)) {
                    //printf("%s",rbuf);
                    break;
                }
                if(strstr((char*)rbuf,"ERROR")) {
                    printf("Expect ERROR: '%s'",rbuf);
                    return -1;
                }
            }
        }
        if (time < 1)
            printf("Startup communications timeout.\n"); 
    }
    return rc;
}

static int expectb(HANDLE hSerial, char byte, char* resp)
{
    int rc;
    int  n = 0;
    char rbuf[MAXRBUF+1];
    char buf[10];   // read 1 char at a time

    rbuf[0] = byte;
    write(hSerial, rbuf, 1);
    waitus(hSerial,10);

    if(strlen(resp) > 0) {
        rbuf[n] = '\0';
        while(1) {
            waitus(hSerial,10);
            rc = read(hSerial,buf,1);
            if(rc < 0) {
                printf("Unexpected port error!\n");
                return rc;
            }
            else if(rc > 0) {
                rbuf[n++] = *buf;
                rbuf[n] = '\0';
                if(strstr((char*)rbuf,resp)) {
                    break;
                }
                if(strstr((char*)rbuf,"ERROR")) {
                    printf("Expect ERROR: '%s'",rbuf);
                    return -1;
                }
            }
        }
    }
    return rc;
}

int erasex  (HANDLE hSerial, char* filename)
{
    int rc = 0;
    //char rbuf[MAXRBUF+1];
    char* mfname = (char*) malloc(strlen(filename)+2);

    flush(hSerial);
    //printf("\nErase  '%s'\n",filename);

    expect(hSerial,erasestr,OKstr);
    sprintf(mfname,"%s\n",filename);
    expect(hSerial,mfname,OKstr);
    free(mfname);
    //printf("Erased '%s'\n",filename);

    return rc;
}

int appex (HANDLE hSerial, char* filename, uint8_t* buf, int count, int sequence)
{
    int rc = count;
    char rbuf[MAXRBUF+1];
    char* mfname = (char*) malloc(strlen(filename)+2);

    flush(hSerial);

    // send append command, expect OK
    expect(hSerial,appe_str,OKstr);

    // send filename, expect OK
    sprintf(mfname,"%s\n",filename);
    expect(hSerial,mfname,OKstr);
    free(mfname);

    // send count, expect OK
    snprintf(rbuf,MAXRBUF,"%d\n",count);
    expect(hSerial,rbuf,OKstr);

    // send bytes, expect # for each one
    //printf("Appending %d bytes.\n", count);
    while(--count > -1) {
        expectb(hSerial,*buf++,"");
    }
    expect(hSerial,"",OKstr);
    printf(".");
    fflush(stdout);
    //printf("\nAppend %s OK [%d]",filename,sequence+1);

    return rc;
}

int savex(HANDLE hSerial)
{
    int rc = 0;
    //char rbuf[MAXRBUF+1];
    expect(hSerial,save_str,OKstr);
    return rc;
}

int nextx (HANDLE hSerial, uint8_t* buf, int count)
{
    int rc = 0;
    char rbuf[MAXRBUF+1];
    expect(hSerial,countstr,OKstr);
    snprintf(rbuf,MAXRBUF,"%d\r",count);
    expect(hSerial,rbuf,OKstr);
    return rc;
}

int resetx  (HANDLE hSerial)
{
    int rc = 0;
    printf("Reboot.\n");
    flush(hSerial);
    expect(hSerial,resetstr,OKstr);
    return rc;
}


/**
 * main entry point
 * @param argc - number of arguments
 * @param argv - array of strings
 * @returns non-zero on error
 */
int main(int argc, char** argv)
{
    int rc = 0;
    int bytes = 0;
    int savesd = 0;
    int savearg = 3;
    int reset = 0;
    int bufsize = 0x2000;
    int type = DOWNLOAD_RUN_BINARY;
    char* port = argv[1];
    char* file = argv[2];
    HANDLE hSerial;
    //DCB sparm;
    FILE* fp = 0;
    int count;
    uint8_t dlbuf[0x8000]; // 32K limit

    if(argc < 2) {
        printf("Usage:\n%s <COMx or /dev/ttyX> <filename> [-r | -s savefile1 savefile2 savefileN]\n", argv[0]);
        printf("Files with .eeprom suffix are handled automatically.\n");
        printf("Use -s write savefile to SD Card.\n");
        printf("Use -r to reset after load complete.\n");
        return 1;
    }

    if(argc > savearg && !strcmp(argv[savearg],"-r")) {
        reset++;
        savearg++;
    }
    if(argc > savearg && !strcmp(argv[savearg],"-s")) {
        savesd++;
        savearg++;
    }


#ifdef WINDOWS
    // strupr the port
    //
    strupr(port);
#endif

    hSerial = openport(port);
    if(hSerial == INVALID_HANDLE_VALUE) {
        printf("Invalid file handle for serial port '%s'\n",port);
    }

    // find propeller
    //
    rc = findprop(hSerial, port);

    // read program if file parameter is available
    //
    if(argc > 1 && file) {
        fp = fopen(file, "rb");
        if(!fp) {
            printf("Can't open '%s' for load.\n", file);
            return 4;
        }
        else {
            count = fread(dlbuf, 1, 0x8000, fp);
            printf("Load Propeller %d bytes of '%s'\n", count, file);
            fclose(fp);
        }
        if(strstr(strupr(file), ".EEPROM") > 0) type = DOWNLOAD_RUN_EEPROM;
    }
    if(!rc && file) rc = upload(hSerial, dlbuf, count, type);
    flush(hSerial);            // take out the garbage

    //printf("Savearg '%d'\n", savearg);

    // if file found, upload file
    //
    if(savesd) {

        if(savearg > 2 && file) {
            char sep = '/';

            sleep(1);             // wait before continuing

            file = argv[savearg];
            if(!rc) {
                fp = fopen(file, "rb");
                if(!fp) {
                    printf("Can't open '%s' for load.\n", file);
                    return 4;
                }
                else {
                    int n = 0;

                    while(strchr(file,sep)) {   // just send filename
                        file = strchr(file,sep)+1;
                        //printf("File '%s'\n", file);
                    }

                    printf("Uploading '%s' ", file);

                    flush(hSerial);            // take out the garbage

                    if(expecttime(hSerial,"?\r",OKstr,10) < 1) {
                        printf("Can't talk to loader. Exiting\n");
                        return 1;   // is it alive?
                    }

                    rc = erasex(hSerial,file); // returns 0 on error
                    //printf("Save   '%s' ", file);

                    while(!feof(fp)) {
                        count = fread(dlbuf, 1, bufsize, fp);
                        rc = appex(hSerial, file, dlbuf, count, n++); // returns num sent
                        bytes += rc;
                        if(!rc) {
                            printf("Save Error at byte '%d'\n", bytes);
                            break;
                        }
                    }
                    fclose(fp);
                }
                if(rc)
                    printf("\n%d bytes saved.\n", bytes);
                rc = 0;
            }
        }
    }


    //printf("Reboot  '%d'\n", reset);
    if(reset)
        resetx(hSerial);

    // close serial port
    closeport(hSerial);

    return rc;
}


