/**
 * @file ComPort.c - a C99 compatible com port driver
 * Copyright (c) 2009 John Steven Denson (jazzed)
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include "inttypes.h"

#include "DebugLib.h"
#include "ComPort.h"

/**
 * open serial port
 * @param port - COMn port name
 * @returns hSerial - file handle to serial port
 */
HANDLE openport(char* port)
{
    HANDLE hSerial;
    DCB sparm;

/*
*/
#ifdef VC2008
    HRESULT hr;

    //Initialize COM (Required by CEnumerateSerial::UsingWMI)
    CoInitialize(NULL);

    //Initialize COM security (Required by CEnumerateSerial::UsingWMI)
    hr = CoInitializeSecurity(NULL, -1, NULL, NULL, RPC_C_AUTHN_LEVEL_DEFAULT, RPC_C_IMP_LEVEL_IMPERSONATE, NULL, EOAC_NONE, NULL);
    if (FAILED(hr)) {
        printf(("Failed to initialize COM security, Error:%x\n"), hr);
        CoUninitialize();
        return 0;  
    }

#endif
    // strupr the port
    //
    strtoupper(port);
    //string sport = sport.Format(_T("\\\\.\\"), port);
#ifdef VC2008
    {
    /*
    char* sport = (char*)malloc(strlen(port)+5);
    extern char* gcomport;
    sprintf(sport, "\\\\.\\%s",port);
    gcomport = sport;
    port = gcomport;
    */
    hSerial = CreateFile(port, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    }
#else
    hSerial = CreateFile(port, GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
#endif
    if(hSerial == INVALID_HANDLE_VALUE) {
        printf("Invalid file handle for serial port '%s'\n",port);
        if(GetLastError()==ERROR_FILE_NOT_FOUND) {
            printf("Can't open serial port '%s'\n",port);
            return 0;
        }
    }

    {
        COMMCONFIG cc;
        DWORD dwSize = sizeof(COMMCONFIG);
        if (!GetDefaultCommConfig(port, &cc, &dwSize)) {
            printf("Error getting serial parms\n");
            CloseHandle(hSerial);
            return 0;
        }
    }
    // set parameters
    //
    sparm.DCBlength=sizeof(sparm);
    if(!GetCommState(hSerial, &sparm)) {
        printf("Error getting serial parms\n");
        CloseHandle(hSerial);
        return 0;
    }
    sparm.BaudRate=CBR_115200;
    sparm.Parity=NOPARITY;
    sparm.StopBits=ONESTOPBIT;
    sparm.ByteSize=8;
    if(!SetCommState(hSerial, &sparm)){
        printf("Error setting serial parms\n");
        CloseHandle(hSerial);
        return 0;
    }
    return hSerial;
}

/**
 * close serial port
 * @param hSerial - file handle to serial port
 */
void closeport(HANDLE hSerial)
{
    CloseHandle(hSerial);
}


/**
 * receive a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to read
 * @returns number of bytes read
 */
int rx(HANDLE hSerial, uint8_t* buff, int n)
{
    DWORD dwBytes = 0;
    if(!ReadFile(hSerial, buff, n, &dwBytes, NULL)){
        printf("Error writing port\n");
        return 0;
    }
    return dwBytes;
}

/**
 * transmit a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to send
 * @returns zero on failure
 */
int tx(HANDLE hSerial, uint8_t* buff, int n)
{
    DWORD dwBytes = 0;
    if(!WriteFile(hSerial, buff, n, &dwBytes, NULL)){
        printf("Error writing port\n");
        return 0;
    }
    return dwBytes;
}

/**
 * hwreset ... resets Propeller hardware using DTR
 * @param hSerial - file handle to serial port
 * @param sparm - pointer to DCB serial control struct
 * @returns void
 */
void hwreset(HANDLE hSerial)
{
    DCB sparm;
    // set parameters
    //
    sparm.DCBlength=sizeof(sparm);
    if(!GetCommState(hSerial, &sparm)) {
        printf("hwreset: Error getting serial parms!\n");
        return;
    }
    sparm.fDtrControl = 1;
    SetCommState(hSerial, &sparm);   // assert DTR to reset
    msleep(25);
    sparm.fDtrControl = 0;
    SetCommState(hSerial, &sparm);   // disassert DTR to start
    msleep(100);
}
