/**
 * @file ComPort.c - a C99 compatible com port driver
 * Copyright (c) 2009 John Steven Denson (jazzed)
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include "inttypes.h"

#include "DebugLib.h"
#include "ComPort.h"

/**
 * open serial port
 * @param port - COMn port name
 * @returns hSerial - file handle to serial port
 */
HANDLE openport(char* port)
{
    HANDLE hSerial;
    DCB sparm;
    COMMTIMEOUTS cto;

    int baud = 0;
    char *rate = strchr(port,':');
    if(rate) {
        port[rate-port] = '\0';
        rate++;
        baud = atoi(rate);
    }

    // strupr the port
    //
    strupr(port);

    hSerial = CreateFile(port, GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if(hSerial == INVALID_HANDLE_VALUE) {
        printf("Invalid file handle for serial port '%s'\n",port);
        if(GetLastError()==ERROR_FILE_NOT_FOUND) {
            printf("No such serial port '%s'\n",port);
            return 0;
        }
    }

    // set parameters
    //
    sparm.DCBlength=sizeof(sparm);
    if(!GetCommState(hSerial, &sparm)) {
        printf("Error getting serial parms\n");
        CloseHandle(hSerial);
        return 0;
    }

    switch(baud) {
        case 0:
            sparm.BaudRate=CBR_115200;
            break;
        case 115200:
            sparm.BaudRate=CBR_115200;
            break;
        case 57600:
            sparm.BaudRate=CBR_57600;
            break;
        case 38400:
            sparm.BaudRate=CBR_38400;
            break;
        default:
            printf("Unsupported baudrate. Use COMn:115200, COMn:57600, or COMn:38400\n");
            closeport(hSerial);
            exit(2);
            break;
    }
    sparm.Parity=NOPARITY;
    sparm.StopBits=ONESTOPBIT;
    sparm.ByteSize=8;
    if(!SetCommState(hSerial, &sparm)){
        printf("Error setting serial parms\n");
        CloseHandle(hSerial);
        return 0;
    }

    //Now we set the timeouts so that ReadFile() will return
    //immediately instead of blocking if there is no data.
    
    cto.ReadIntervalTimeout = MAXDWORD;
    cto.ReadTotalTimeoutMultiplier = 0;
    cto.ReadTotalTimeoutConstant = 0;
    cto.WriteTotalTimeoutMultiplier = 0;
    cto.WriteTotalTimeoutConstant = 0;
    if(SetCommTimeouts(hSerial, &cto) == 0) {
        CloseHandle(hSerial);
        printf("Couldn't set %s port timeouts.", port);
        return 0;
    }

    return hSerial;
}

/**
 * close serial port
 * @param hSerial - file handle to serial port
 */
void closeport(HANDLE hSerial)
{
    CloseHandle(hSerial);
}


/**
 * receive a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to read
 * @returns number of bytes read
 */
int rx(HANDLE hSerial, uint8_t* buff, int n)
{
    DWORD dwBytes = 0;
    if(!ReadFile(hSerial, buff, n, &dwBytes, NULL)){
        printf("Error reading port\n");
        return 0;
    }
    return dwBytes;
}

/**
 * transmit a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to send
 * @returns zero on failure
 */
int tx(HANDLE hSerial, uint8_t* buff, int n)
{
    DWORD dwBytes = 0;
    if(!WriteFile(hSerial, buff, n, &dwBytes, NULL)){
        printf("Error reading port\n");
        return 0;
    }
    return dwBytes;
}

/**
 * hwreset ... resets Propeller hardware using DTR
 * @param hSerial - file handle to serial port
 * @param sparm - pointer to DCB serial control struct
 * @returns void
 */
void hwreset(HANDLE hSerial)
{
    DCB sparm;
    // set parameters
    //
    sparm.DCBlength=sizeof(sparm);
    if(!GetCommState(hSerial, &sparm)) {
        printf("hwreset: Error getting serial parms!\n");
        return;
    }
    sparm.fDtrControl = 1;
    SetCommState(hSerial, &sparm);   // assert DTR to reset
    msleep(25);
    sparm.fDtrControl = 0;
    SetCommState(hSerial, &sparm);   // disassert DTR to start
    msleep(100);
}
