/**
 * @file DebugLib.h - client debug library
 * Copyright (c) 2009 John Steven Denson (jazzed)
 * All Rights Reserved.
 */
#ifndef DEBUGLIB_H
#define DEBUGLIB_H

int init (void);

/**
 * sleep for ms milliseconds
 * @param ms - num ms to sleep
 */
void msleep(int ms);

/**
 * like strlower
 */
char* strtolower(char* s);

/**
 * like strupper
 */
char* strtoupper(char* s);

/**
 * get rid of path stuff
 */
char* killPath(char* file);

/**
 * build spin binary
 * @param compcmd - compiler command to use
 * @param fname - filename to build ... contains path info
 * @param opts - compile options
 * @param libp - compiler library path
 * @returns non-zero on failure
 */
int buildSpin(char* compcmd, char *fname, char* opts, char* libp);

/**
 * load spin binary
 * @param loader - loader command to use
 * @param fname - filename to load ... contains path info
 * @param opts - load options ... just COMn port for now
 * @returns non-zero on failure
 */
int loadSpin(char* loader, char *fname, char* opts);

/**
 * build and reload spin program using buildSpin, loadSpin and globals
 * @returns non-zero on failure
 */
int reload(void);

/**
 * trim whitespace off of a string
 */
char* trim(char* s);

/**
 * check if the string and #checksum match
 * @param str - string to check
 * @param returns non-zero if ok
 */
int chksumOk (char *str);

/**
 * open a file for read with filename
 * @param name - file name to open
 * @returns FILE* struct
 */
FILE* openfile(char *name);

/**
 * find the line number containing symbol from start line in +/- direction
 * @param fp - previously opened FILE* handle
 * @param symbol - symbol to find
 * @param sline - start line for finding symbol
 * @param direction - direction to search for finding symbol
 * @returns line number where symbol is found
 */
int   findline(FILE* fp, char *symbol, int sline, int direction);

/**
 * find the char position number containing symbol from start position in +/- direction
 * @param fp - previously opened FILE* handle
 * @param symbol - symbol to find
 * @param start - start position for finding symbol
 * @param direction - direction to search for finding symbol
 * @returns char position number where symbol is found
 */
int   findstart(FILE* fp, char *symbol, int start, int direction);

/**
 * read line from file at line number into buff
 * @param fp - previously opened FILE* handle
 * @param line - line to read
 * @param buff - buffer where line is placed
 * @param blen - maximum length of buff
 * @returns line or zero on error
 */
int   readline (FILE* fp, int line, char *buff, int blen);

/**
 * read from file name count lines of text from start line into buff
 * @param name - file name to read
 * @param start - start line to read
 * @param count - number of lines to read
 * @param buff - buffer where lines are placed
 * @param blen - maximum length of buff
 * @returns char position number where symbol is found
 */
int   readlines(char *name, int start, int count, char *buff, int blen);


#endif
//DEBUGLIB_H
