/**
 * @file DebugLib.c - client debug library
 * Copyright (c) 2009 John Steven Denson (jazzed)
 * All Rights Reserved.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/timeb.h>

#include "inttypes.h"
#include "ComPort.h"
#include "DebugLib.h"

#ifdef WINDOWS
#include <windows.h>

static unsigned long getms()
{
    LARGE_INTEGER ticksPerSecond;
    LARGE_INTEGER tick;   // A point in time
    LARGE_INTEGER time;   // For converting tick into real time
    // get the high resolution counter's accuracy
    QueryPerformanceFrequency(&ticksPerSecond);
    if(ticksPerSecond.QuadPart < 1000) {
        printf("Your system does not meet timer requirement. Try another computer. Exiting program.\n");
        exit(1);
    }
    // what time is it?
    QueryPerformanceCounter(&tick);
    time.QuadPart = (tick.QuadPart*1000/ticksPerSecond.QuadPart);
    return (unsigned long)(time.QuadPart);
}

static void waitms(int ms)
{
    unsigned long t = getms();
    while((t+ms+10) > getms())
        ;
}
#else
static volatile struct timeb t0, t1;
#endif

/**
 * sleep for ms milliseconds
 * @param ms - time to wait in milliseconds
 */
void msleep (int ms)
{
#ifdef WINDOWS
    waitms(ms);
#else
    while(ms-- > 0) {
        ftime((struct timeb*)&t0);
        do {
            ftime((struct timeb*)&t1);
        } while (t1.millitm == t0.millitm);
    }
#endif
}

char* strtolower(char* s)
{
    int len = strlen(s);
    while(--len >= 0)
        s[len] = tolower(s[len]);
    return s;
}

char* strtoupper(char* s)
{
    int len = strlen(s);
    while(--len >= 0)
        s[len] = toupper(s[len]);
    return s;
}


char* strchrr(char* haystack, char needle)
{
    int len = strlen(haystack)-1;
    while(len >= 0) {
        if(haystack[len] == needle)
            return &haystack[len];
        len--;
    }
    return 0;
}

/**
 * save .list filename
 */
void  saveListFileName(char* fname)
{
    int rc = 0;
    strcpy(gfilepath,fname);

    rc = strchrr(gfilepath,*PATHSEP)-gfilepath+1;
    if(rc > 0)
        gfilepath[rc] = '\0';
    else
        sprintf(gfilepath,".%c",*PATHSEP);
    printf("Top Spin File Path: '%s'\n",gfilepath);

    sprintf(gtoplist,"%s.list",fname);
    //strcpy(gtoplist,fname);
    //gtoplist[strlen(fname)] = '\0';
    //strcat(gtoplist,".list");
    printf("Top List File: '%s'\n",gtoplist);
}

/**
 * loader
 */
int loadSpin (char* loader, char *fname, char* opts)
{
    extern int pload(char* file, char* port); // bad, lazy programmer

    char binfile[BUFLEN];
    int   sz = (int)(strstr(fname,".spin")-fname+1);
    strcpy(binfile,fname);
    binfile[sz] = '\0';
    strcat(binfile,"binary");
    return pload(binfile, opts);
}

/**
 * reloader
 */
int reload (void)
{
    if(*gloaderfile != 0 && *gcomport != 0) {
        if(loadSpin(gloaderfile, gtopfile, gcomport) != 0)
            return 2;
    }
    else
        return 4 | 2;

    return 0;
}

/**
 * trim whitespace off of a string
 */
char* trim (char* s)
{
    int len = strlen(s);
    while(isspace(*s))
        s++;
    while(len > 0 && isspace(s[len-1]))
        len--;
    s[len] = '\0';
    return s;
}

/**
 * chksumOk
 */
int chksumOk (char *str)
{
    int sum = 0;
    int sumv = 0;
    int n = 0;
    int len = strlen(str);
    char *p;
    // do first to last incase we get a second message
    for(n = 0; n < len && str[n] != '#'; n++) {
        sum = (sum + str[n]) & 0xff;
    }
    p = &str[++n];
    p[2] = '\0';
    sumv = strtol(p,0,16);
    /*
    while(n > 0 && str[n-1] != '#') 
        n--;
    if(n <= 0)
        return 0;
    sumv = strtol(trim(&str[n]),0,16);
    n--; // go below the #
    while(--n >= 0) {
        sum = (sum + str[n]) & 0xff;
    }
    */
    //if(sum != sumv) printf("Bad Sum\n");
    return sum == sumv ? 1 : 0;
}

/**
 * open file name for read
 */
FILE* openfile (char *name)
{
    FILE *p = fopen(name, "r");
    if(p == 0) {
        char *filename = (char*)malloc(strlen(name)+1+strlen(glibrary)+1);
        char *np = strrchr(name,*PATHSEP);
        if(np) {
            sprintf(filename,"%s%s",glibrary,np);
            p = fopen(filename,"r");
            free(filename);
        }
    }
    if(p == 0) {
        printf("Cant find file '%s'.\n", name);
        return 0;
    }
    return p;
}

/**
 * find line of symbol in file from sline (1 based) in +/- direction
 */
int   findline (FILE* fp, char *symbol, int sline, int direction)
{
    int rc = 0;
    int lino = 0;
    char buff[BUFLEN];

    // rewind file
    // ... if direction < 0, fgets match and return last match on line match
    // ... if direction > 0, fgets to line, then return first match
    
    // go to start line
    rewind(fp);

    // fetch lines
    while(!feof(fp)) {
        fgets(buff, BUFLEN, fp);    // get line
        lino++;                     // count lines
        if(direction > 0 && lino <  sline) // don't comp until sline
            continue;
        if(strstr(buff, symbol) != 0) {
            rc = lino;              // got something
            if(direction > 0 && lino >= sline)       // break if direction > 0
                break;
        }
        if(direction < 0 && lino >= sline)
            break;                  // if linematch and direction < 0, break
    }
    return rc;
}

/**
 * find start position of symbol in file from char start in +/- direction
 */
int   findstart (FILE* fp, char *symbol, int start, int direction)
{
    int rc = 0;
    int pos = 0;
    char buff[BUFLEN];

    // go to start position
    rewind(fp);

    // fetch lines
    while(!feof(fp)) {
        fgets(buff, BUFLEN, fp); // get line & position
        pos += strlen(buff);
        if(direction > 0)               // don't comp until sline
            continue;
        if(strstr(buff, symbol) != 0)   // got something
            rc = pos + strstr(buff,symbol)-buff;
        if(direction < 0 && pos == start)
            break;                      // if pos match and direction < 0, break
    }
    return rc;
}

/**
 * count number of c's in sp
 */
static int count(char* sp, char c)
{
    int rc = 0;
    int len = strlen(sp);
    while(len-- > 0)
        if(*(sp++) == c)
            rc++;
    return rc;
}

/**
 * get line terminated by '\r'
 */
static int getCrLine(char* sp, int num)
{
    int rc = 0;
    int n = 0;
    int eol = 0;
    int len = strlen(sp);
    char *op = sp;
    for(n = 0; sp[n] && n < len; n++) {
        if(sp[n] == '\r') {
            rc++;
            if(rc == num) {
                if(strchr(op,'\r') != 0) {
                    eol = strchr(op,'\r')-op;
                    op[eol+1] = '\n';
                    op[eol+2] = '\0';
                }
                strcpy(sp,op);
                break;
            }
            if(n+1 < len)
                op = &sp[n+1];
        }
    }
    return rc;
}

/**
 * read one line from file from start line (1 based) into buff
 */
int   readline (FILE* fp, int line, char *buff, int blen)
{
    int rc = 0;
    int lino = 0;
    int n = 0;
    char rdbuf[512];

    if(line < 0) return rc;             // not valid
    
    // go to start line
    rewind(fp);

    // fetch lines
    // some files may contain '\r' only for end of line ... must be "think different" :)
    // ... what a fucking pain in the ass.
    while(!feof(fp)) {
        fgets(rdbuf, 512, fp);          // get line
        n = count(rdbuf,'\r');          // if multiple '\r', then not a normal file
        if(n > 1) {
            if(lino+n >= line) {        // if line number match, get line
                getCrLine(rdbuf,line-lino);
                strcpy(buff, rdbuf);
                rc = line;
                break;
            }
            lino += n;
            fseek(fp,-1,SEEK_CUR);      // go back if we get too much
            while(fgetc(fp) != '\r')
                fseek(fp,-2,SEEK_CUR);
        }
        else {
            lino++;
            if(lino == line) {          // count lines
                strcpy(buff, rdbuf);
                rc = line;
                break;                  // if linematch and direction < 0, break
            }
        }
    }
    return rc;
}

/**
 * read from file name count lines of text from start line into buff
 */
int   readlines (char *name, int start, int count, char *buff, int blen)
{
    return 0;
}

char* killPath(char* file)
{
    int len = strlen(file);
    char* sp = file+len;
    while(*sp != *PATHSEP && len--)
        sp--;
    if(len > 0)
        file = sp+1;
    return file;
}

