/**
 * @file ComPort.c - a C99 compatible com port driver
 * Copyright (c) 2009 John Steven Denson (jazzed)
 * All Rights Reserved.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/select.h>
#include "inttypes.h"

#include "DebugLib.h"
#include "ComPort.h"

/**
 * open serial port
 * @param port - COMn port name
 * @returns hSerial - file handle to serial port
 */
HANDLE openport(char* port)
{
    HANDLE hSerial;
    struct termios sparm;
    int tbaud = 0;
    int baud = 0;
    char *rate = strchr(port,':');
    if(rate) {
        port[rate-port] = '\0';
        rate++;
        baud = atoi(rate);
    }

    hSerial = open(port, O_RDWR | O_NOCTTY);
    if(hSerial == -1) {
        printf("Invalid file handle for serial port '%s'\n",port);
        printf("Error '%s'\n",strerror(errno));
        return 0;
    }

    switch(baud) {
        case 0:
            tbaud = B115200;
            break;
        case 115200:
            tbaud = B115200;
            break;
        case 57600:
            tbaud = B57600;
            break;
        case 38400:
            tbaud = B38400;
            break;
        default:
            printf("Unsupported baudrate. Use /dev/tty*:115200, /dev/tty*:57600, or /dev/tty*:38400\n");
            closeport(hSerial);
            exit(2);
            break;
    }

    // set parameters
    bzero(&sparm, sizeof(sparm));
    // 115200 bps, n18, local connection, enable receive
    sparm.c_cflag = tbaud | CS8 | CLOCAL | CREAD;
    // ignore parity | map CR to NL
    sparm.c_iflag = IGNPAR | ICRNL;
    // raw output
    sparm.c_oflag = 0;
    // disable echo
    sparm.c_lflag = 0; // no ICANON;

    tcflush(hSerial, TCIFLUSH);
    tcsetattr(hSerial, TCSANOW, &sparm);

    return hSerial;
}

/**
 * close serial port
 * @param hSerial - file handle to serial port
 */
void closeport(HANDLE hSerial)
{
    close(hSerial);
}

/**
 * receive a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to read
 * @returns number of bytes read
 */
int rx(HANDLE hSerial, uint8_t* buff, int n)
{
    ssize_t bytes = read(hSerial, buff, n);
    if(bytes < 1) {
        //printf("Error reading port\n");
        return 0;
    }
    return bytes;
}

/**
 * transmit a buffer
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to send
 * @returns zero on failure
 */
int tx(HANDLE hSerial, uint8_t* buff, int n)
{
    ssize_t bytes = write(hSerial, buff, n);
    if(bytes != n) {
        printf("Error writing port\n");
        return 0;
    }
    return bytes;
}

/**
 * transmit a buffer and wait for response
 * @param hSerial - file handle to serial port
 * @param buff - char pointer to buffer
 * @param n - number of bytes in buffer to send
 * @returns zero on failure
 */
int txwu(HANDLE hSerial, uint8_t* buff, int n, int waitu)
{
    struct timeval tv;
    ssize_t bytes;
    bytes = write(hSerial, buff, n);
    if(bytes != n) {
        printf("Error writing port\n");
        return 0;
    }
    tv.tv_sec  = 0;
    tv.tv_usec = waitu;
    select(0,0,0,0,&tv);
    return bytes;
}

/**
 * hwreset ... resets Propeller hardware using DTR
 * @param hSerial - file handle to serial port
 * @param sparm - pointer to DCB serial control struct
 * @returns void
 */
void hwreset(HANDLE hSerial)
{
    int cmd = TIOCM_DTR;
    ioctl(hSerial, TIOCMBIS, &cmd); // assert DTR pin
    msleep(25);
    ioctl(hSerial, TIOCMBIC, &cmd); // deassert DTR pin
    msleep(100);
}
