/**
 * @file tvText.h
 * tvText native device driver interface
 *
 * Copyright (c) 2010, Steve Denson
 * See end of file for terms of use.
 */
 
/**
 * This module uses the TV_Half_Height.spin driver converted to a .o
 */
 
#ifndef __TV2TEXT__
#define __TV2TEXT__

/**
 * TV_Text color indicies
 */
#define TV_TEXT_WHITE_BLACK     0
#define TV_TEXT_WHITE_BLUE      1
#define TV_TEXT_WHITE_RED       2
#define TV_TEXT_YELLOW_BROWN    3
#define TV_TEXT_GREY_WHITE      4
#define TV_TEXT_CYAN_BLUE       5
#define TV_TEXT_GREEN_GREYGREEN 6
#define TV_TEXT_RED_PINK        7

#define TV_TEXT_COLORS 16

/**
 * Default TV_Text palette color indicies
 */
#define TV_TEXT_PAL_WHITE_BLACK     0
#define TV_TEXT_PAL_WHITE_BLUE      2
#define TV_TEXT_PAL_WHITE_RED       4
#define TV_TEXT_PAL_YELLOW_BROWN    6
#define TV_TEXT_PAL_GREY_WHITE      8
#define TV_TEXT_PAL_CYAN_BLUE       10
#define TV_TEXT_PAL_GREEN_GREYGREEN 12
#define TV_TEXT_PAL_RED_PINK        14

/**
 * tvText colors
 */
#define BACKGROUND_COLOR DARK_GREY
#define FOREGROUND_COLOR INDIAN_RED4

#define BOXEFX_BRIGHT   7
#define BOXEFX_LIGHT    5
#define BOXEFX_DARKEST  1
#define BOXEFX_DARK     2
#define BOXEFX_NONE     4

#define BLACK           1
#define DARKEST_GREY    2
#define DARK_GREY       3
#define GREY            4
#define LIGHT_GREY      5
#define WHITE_GREY      6
#define WHITE           7
#define INDIAN_RED4     185
#define CORAL3          170
#define CORAL2          171
#define SANDY_BROWN     172
#define LIGHT_SALMON2   190
#define YELLOW          155
#define BROWN           158
#define WHITE2          103
#define LIGHT_GREEN     8
#define BLUE            9
#define BLUE2           10
#define VIOLET          11
#define PURPLE          12
#define PURPLE3         250
#define MED_PURPLE4     233
#define LT_PURPLE       13
#define LT_PURPLE1      14
#define LT_GREEN        15
#define L1_GREEN        17
#define GOLD            24
#define GOLD2           173
#define GOLD3           157
#define DODGER_BLUE     28
#define DODGER_BLUE2    175
#define AZURE           29
#define SKY_BLUE        29
#define LIGHT_SKY_BLUE  30
#define GOLDENROD       31
#define KHAKI1          174
#define DARK_GOLDENROD4 154
#define LIGHT_GOLDENROD4 156
#define GOLDENROD2      189
#define GOLDENROD4      155
#define BURNT_ORANGE    40
#define ORANGE          188
#define DARK_ORANGE2    188
#define CORNFLOWER_BLUE 41
#define ROYAL_BLUE      43
#define ROYAL_BLUE2     152
#define DEEP_SKY_BLUE   44
#define STEEL_BLUE      45
#define STEEL_BLUE1     168
#define STEEL_BLUE2     191
#define LIGHT_CYAN      46
#define CYAN            46
#define RED             0xBB
#define FIREBRICK       47
#define FIREBRICK2      56
#define FIREBRICK4      202
#define MAROON          203
#define MAROON2         204
#define MAGENTAX        205
#define PINK            206
#define NAVY            57
#define NAVY_BLUE       57
#define SLATE_BLUE      59
#define CYAN1           61
#define CYAN2           62
#define CYAN0           60
#define FIREBRICK3      63
#define HOT_PINK        72
#define HOT_PINK4       217
#define HOT_PINK1       111
#define HOT_PINK2       120
#define LIGHT_PINK      222
#define LIGHT_PINK1     223
#define SEA_GREEN4      74
#define FOREST_GREEN    75
#define SEA_GREEN       76
#define TURQUOISE       77
#define TURQUOISE1      200
#define TURQUOISE2      78
#define MAGENTA         88
#define MAGENTA1        104
#define MAGENTA2        95
#define DARK_SLATE_GREY 89
#define DARK_GREEN      90
#define GREEN           107
#define GREEN1          108
#define GREEN2          109
#define GREY_GREEN      110
#define GREYGRN         110
#define YELLOW_GREEN    92

/**
 * TV_Text color table size.
 * Table holds foreground and background info, so size is 2 x table colors.
 */
#define TV_TEXT_COLORTABLE_SIZE 4*8*2   // 64 for TV_Half_Height

/**
 * TV_Text column count
 */
#define  TV_TEXT_COLS 44

/**
 * TV_Text row count
 */
#define  TV_TEXT_ROWS 54

/**
 * TV_Text screensize count
 */
#define  TV_TEXT_SCREENSIZE (TV_TEXT_COLS * TV_TEXT_ROWS)

/**
 * TV_Text lastrow position count
 */
#define  TV_TEXT_LASTROW (TV_TEXT_SCREENSIZE-TV_TEXT_COLS*2)

/**
 * TV_Text status enum
 */
typedef enum {
    TV_TEXT_STAT_DISABLED,
    TV_TEXT_STAT_INVISIBLE,
    TV_TEXT_STAT_VISIBLE
} TvTextStat_t;

/**
 * TV_Text control struct
 */
typedef struct _tv_text_struct
{
    uint32_t status     ; //0/1/2 = off/invisible/visible              read-only
    uint32_t enable     ; //0/non-0 = off/on                           write-only
    uint32_t pins       ; //%pppmmmm = pin group, pin group mode       write-only
    uint32_t mode       ; //%tccip = tile,chroma,interlace,ntsc/pal    write-only
    uint16_t *screen    ; //pointer to screen (words)                  write-only      
    uint32_t *colors    ; //pointer to colors (longs)                  write-only
    uint32_t ht         ; //horizontal tiles                           write-only
    uint32_t vt         ; //vertical tiles                             write-only
    uint32_t hx         ; //horizontal tile expansion                  write-only
    uint32_t vx         ; //vertical tile expansion                    write-only
    uint32_t ho         ; //horizontal offset                          write-only
    uint32_t vo         ; //vertical offset                            write-only
    uint32_t broadcast  ; //broadcast frequency (Hz)                   write-only
    uint32_t auralcog   ; //aural fm cog                               write-only      
}   TvText_t;

/*
 * TV_Text public API
 */

/**
 * TV_Text start function starts TV on a cog
 * @param basepin is first pin number (out of 8) connected to TV
 * param clockrate is the clockrate defined for the platform.
 * @returns non-zero cogid on success
 */
int     tvText_start(int basepin);

/**
 * TV_Text stop function stops TV cog
 */
void    tvText_stop(void);

/**
 * TV_Text str function prints a string at current position
 * @param sptr is string to print
 */
void    tvText_str(char* sptr);

/**
 * TV_Text dec function prints a decimal number at current position
 * @param value is number to print
 */
void    tvText_dec(int value);

/**
 * TV_Text hex function prints a hexadecimal number at current position
 * @param value is number to print
 * @param digits is number of digits in value to print
 */
void    tvText_hex(int value, int digits);

/**
 * TV_Text bin function prints a binary number at current position
 * @param value is number to print
 * @param digits is number of digits in value to print
 */
void    tvText_bin(int value, int digits);

/**
 * TV_Text out function prints a character at current position or performs
 * a screen function based on the following table:
 *
 *    $00 = clear screen
 *    $01 = home
 *    $08 = backspace
 *    $09 = tab (8 spaces per)
 *    $0A = set X position (X follows)
 *    $0B = set Y position (Y follows)
 *    $0C = set color (color follows)
 *    $0D = return
 * others = printable characters
 *
 * @param value is number to print
 * @param digits is number of digits in value to print
 */
void    tvText_out(int c);

/**
 * TV_Text setcolors function sets the palette to that defined by pointer.
 *
 * Override default color palette
 * palette must point to a list of up to 8 colors
 * arranged as follows (where r, g, b are 0..3):
 *
 *               fore   back
 *               ------------
 * palette  byte %%rgb, %%rgb     'color 0
 *          byte %%rgb, %%rgb     'color 1
 *          byte %%rgb, %%rgb     'color 2
 *          ...
 *
 * @param palette is a char array[16].
 */
void    tvText_setColorPalette(char* palette);

/*
 * TV_Text getTile gets tile data from x,y position
 * See header file for more details.
 */
uint16_t tvText_getTile(int x, int y);

/*
 * TV_Text setTile sets tile data at x,y position
 * @param x is screen column position
 * @param y is screen row position
 * @param tile is tile index
 */
void tvText_setTile(int x, int y, uint16_t tile);

/*
 * TV_Text setTileColor sets tile data color at x,y position
 * @param x is screen column position
 * @param y is screen row position
 * @returns tile color palette index
 */
uint16_t tvText_getTileColor(int x, int y);

/*
 * TV_Text setTileColor sets tile data color at x,y position
 * @param x is screen column position
 * @param y is screen row position
 * @param tile color palette index
 */
void tvText_setTileColor(int x, int y, uint16_t color);

/**
 * TV_Text setCurPositon function sets position to x,y.
 * @param x is column counted from left.
 * @param y is row counted from top.
 */
void    tvText_setCurPosition(int x, int y);

/**
 * TV_Text setCoordPosition function sets position to cartesian x,y.
 * @param x is column counted from left.
 * @param y is row counted from bottom.
 */
void    tvText_setCoordPosition(int x, int y);

/**
 * TV_Text setXY function sets position to x,y.
 * @param x is column counted from left.
 * @param y is row counted from top.
 */
void    tvText_setXY(int x, int y);

/**
 * TV_Text setX function sets column position value
 * @param value is new column position
 */
void    tvText_setX(int value);

/**
 * TV_Text setY function sets row position value
 * @param value is new row position
 */
void    tvText_setY(int value);

/*
 * TV_Text setYhalf function sets half row position value
 * @param value is new half row position
 */
void    tvText_setYhalf(int value);

/**
 * TV_Text getX function gets column position
 * @returns column position
 */
int tvText_getX(void);

/**
 * TV_Text getY function gets row position
 * @returns row position
 */
int tvText_getY(void);

/**
 * TV_Text setColors function sets palette color set index
 * @param value is a color set index number 0 .. 7
 */
void tvText_setColors(int value);

/**
 * TV_Text getColors function gets palette color set index
 * @returns number representing color set index
 */
int tvText_getColors(void);

/**
 * TV_Text getWidth function gets screen width.
 * @returns screen column count.
 */
int tvText_getColumns(void);

/**
 * TV_Text getHeight function gets screen height.
 * @returns screen row count.
 */
int tvText_getRows(void);

/**
 * TV_Text print null terminated char* to screen with normal stdio definitions
 * @param s is null terminated string to print using putchar
 */
void    print(char* s);

/**
 * TV_Text putchar print char to screen with normal stdio definitions
 * @param c is character to print
 */
// void    putchar(char c); // let stdio.h define

/**
 * tvText_button draws a button of width/height/color at left/top with text.
 *
 * @param left   - left side button position
 * @param top    - top  side button position
 * @param width  - width of button
 * @param height - height of button
 * @param color  - button color
 * @param text   - button text
 */
void tvText_button(int left, int top, int width, int height, int color, char* text);

/**
 * tvText_box draws a box of width/height/color at left/top.
 *
 * @param left   - left side box position
 * @param top    - top  side box position
 * @param width  - width of box
 * @param height - height of box
 * @param color  - box color
 */
void tvText_box(int left, int top, int width, int height, int color);

/**
 * tvText_rectangle draws a rectangle of width/height/color at left/top.
 *
 * @param left   - left side rectangle position
 * @param top    - top  side rectangle position
 * @param width  - width of rectangle
 * @param height - height of rectangle
 * @param color  - rectangle color
 * @param fill   - fill rectangle with color if non-zero
 */
void tvText_rectangle(int left, int top, int width, int height, int color, int fill);

#endif
//__TV_TEXT__

/*
+------------------------------------------------------------------------------------------------------------------------------+
                                                   TERMS OF USE: MIT License                                                                                                              
+------------------------------------------------------------------------------------------------------------------------------
Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation     
files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
is furnished to do so, subject to the following conditions:                                                                   
                                                                                                                              
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
                                                                                                                              
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
+------------------------------------------------------------------------------------------------------------------------------+
*/

