/**
 * @file tv_text.c
 * TV_Text native device driver interface.
 *
 * Copyright (c) 2008, Steve Denson
 * See end of file for terms of use.
 */
#include <ctype.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include <propeller.h>
#include <Tv2Text.h>

#define TV_TEXT_OUT

// forward declarations

void initColors();

/**
 * This is the main global tv text control/status structure.
 */
volatile TvText_t *tvPtr;

/**
 * These are variables to keep up with display;
 */
static int col, row, flag;

static uint16_t blank = 0x220;

/**
 * This is the TV palette.
 */
static uint8_t gpalette[TV_TEXT_COLORTABLE_SIZE] =     
{
    //       fore            back
    //       color           color
    WHITE,          BLACK,  //0    white / black
    WHITE,          BLUE2,  //1    white / dark blue
    WHITE,          RED,    //2    white / red
    YELLOW,         BROWN,  //3   yellow / brown
    DARK_GREY,      WHITE,  //4     grey / white
    CYAN,           BLUE2,  //5     cyan / blue
    GREEN,          GREYGRN,//6    green / gray-green
    RED,            PINK,   //7      red / pink

    WHITE,          LIGHT_GREY,   //8    white / grey
    WHITE,          BLUE2,  //9    white / dark blue
    WHITE,          RED,    //10   white / red
    YELLOW,         BROWN,  //11  yellow / brown
    GREY,           WHITE,  //12    grey / white
    CYAN,           BLUE2,  //13    cyan / blue
    GREEN,          GREYGRN,//14   green / gray-green
    RED,            PINK,   //15     red / pink

    //       highlight       buttonface
    //       color           color
    CYAN,           BLUE,   //0
    YELLOW,         BLUE,   //1
    GOLD,           AZURE,  //2
    WHITE,          NAVY,   //3
    STEEL_BLUE,     NAVY,   //4
    PURPLE,         BLUE2,  //5
    PURPLE,         BLUE2,  //6
    PURPLE,         BLUE2,  //7

    CYAN,           BLUE,   //0
    YELLOW,         BLUE,   //1
    GOLD,           AZURE,  //2
    WHITE,          NAVY,   //3
    STEEL_BLUE,     NAVY,   //4
    PURPLE,         BLUE2,  //5
    PURPLE,         BLUE2,  //6
    PURPLE,         BLUE2   //7
};

/*
 * This should set the character foreground and screen background.
 * API are available to get/set this.
 */
static int gcolor = 0;

/*
 * global var to keep cogid so we don't have to pass parm to stop
 */
static int gTvTextCog = 0;

/*
 * delay function used after start cog and during screen updates
 * @param per is number of ticks to wait ....
 */
static void wait(int per)
{
    // not the best way to do it !
    volatile int time = 0;
    for(time = 0; time < per; )
        time++;
}    

static inline void wordfill(uint16_t *dst, uint16_t val, int len)
{
    while(--len > -1) {
        *dst++ = val;
    }
}

/*
 * TV_Text start function starts TV on a cog
 * See header file for more details.
 */
int     tvText_start(int basepin)
{
    extern int _binary_TV_Half_Height_dat_start;
    extern int _binary_TV_Half_Height_dat_end;

    int     error   = 0;
    void    *pasm   = 0;

    int     binsize = &_binary_TV_Half_Height_dat_end - &_binary_TV_Half_Height_dat_start;

    col     = 0;
    row     = 0;
    flag    = 0;

    tvPtr   = (TvText_t*) _syscall(&error,SYS_huballoc,sizeof(TvText_t));
    pasm    = (uint32_t*) _syscall(&error,SYS_huballoc,binsize*4);
    longmove(pasm,&_binary_TV_Half_Height_dat_start,binsize);

    tvPtr->status = 0;
    tvPtr->enable = 1;
    tvPtr->pins   = ((basepin & 0x38) << 1) | (((basepin & 4) == 4) ? 0x5 : 0);
    tvPtr->mode   = 0x02;
    tvPtr->colors = (uint32_t*) _syscall(&error,SYS_huballoc,TV_TEXT_COLORTABLE_SIZE);
    tvPtr->screen = (uint16_t*) _syscall(&error,SYS_huballoc,TV_TEXT_SCREENSIZE*2);
    tvPtr->ht = TV_TEXT_COLS;
    tvPtr->vt = TV_TEXT_ROWS;
    tvPtr->hx = 4;
    tvPtr->vx = 1;
    tvPtr->ho = 0;
    tvPtr->vo = 0;
    tvPtr->broadcast = 0;
    tvPtr->auralcog  = 0;

#ifdef DEBUG
    iprintf("pasm   %08x\n", (uint32_t)pasm);
    iprintf("tvPtr  %08x\n", (uint32_t)tvPtr);

    iprintf("stat   %08x %08x\n", &tvPtr->status, tvPtr->status);
    iprintf("enable %08x %08x\n", &tvPtr->enable, tvPtr->enable);
    iprintf("pins   %08x %08x\n", &tvPtr->pins, tvPtr->pins);
    iprintf("mode   %08x %08x\n", &tvPtr->mode, tvPtr->mode);
    iprintf("screen %08x %08x\n", &tvPtr->screen, tvPtr->screen);
    iprintf("colors %08x %08x\n", &tvPtr->colors, tvPtr->colors);
    iprintf("ht     %08x %08x\n", &tvPtr->ht, tvPtr->ht);
    iprintf("vt     %08x %08x\n", &tvPtr->vt, tvPtr->vt);
    iprintf("hx     %08x %08x\n", &tvPtr->hx, tvPtr->hx);
    iprintf("vx     %08x %08x\n", &tvPtr->vx, tvPtr->vx);
    iprintf("ho     %08x %08x\n", &tvPtr->ho, tvPtr->ho);
    iprintf("vo     %08x %08x\n", &tvPtr->vo, tvPtr->vo);
    iprintf("broad  %08x %08x\n", &tvPtr->broadcast, tvPtr->broadcast);
    iprintf("aural  %08x %08x\n", &tvPtr->auralcog, tvPtr->auralcog);

    iprintf("*stat  %08x\n", tvPtr->status);
#endif

    initColors();

    // set main fg/bg color
    tvText_setColorPalette(&gpalette[2]);
    gcolor = 0;

    // blank the screen
    wordfill(tvPtr->screen, blank, TV_TEXT_SCREENSIZE);

    // start new cog from external memory using pasm and tvPtr
    gTvTextCog = cognew(pasm, (void*)tvPtr) + 1;

    wait(1000); // wait for COG start

    _syscall(&error,SYS_hubfree,pasm);  // free pasm space
    
    tvText_out(0);
    //tvText_str("Hello World!\n");

/*
    int     n       = 0;
    for(n = 0; n < TV_TEXT_SCREENSIZE; n++) {
        if((n % 16) == 0)
            iprintf("\n%08x:", &tvPtr->screen[n]);
        iprintf(" %04x", tvPtr->screen[n]);
    }
    iprintf("\n\n");
*/
    return gTvTextCog;
}

/**
 * stop stops the cog running the native assembly driver 
 */
void tvText_stop(void)
{
    int id = gTvTextCog - 1;
    if(gTvTextCog > 0) {
        cogstop(id);
    }
}
/*
 * TV_Text setcolors function sets the palette to that defined by pointer.
 * See header file for more details.
 */
void    tvText_setColorPalette(char* ptr)
{
    int  ii, n;
    uint8_t  fg = 0;
    uint8_t  bg = 0;

    uint32_t* colors = tvPtr->colors;

    for(ii = 0; ii < TV_TEXT_COLORS; ii++)
    {
        n = ii << 1;
        fg = (uint8_t) ptr[n];
        bg = (uint8_t) ptr[n+1];
        colors[n]     = fg << 24 | bg << 16 | fg << 8 | bg;
        colors[n+1]   = fg << 24 | fg << 16 | bg << 8 | bg;
   }        
}

// Build an indexable color table

static uint32_t colorset[64];

// This routine builds a sane color set that we can index

void initColors()
{
    int n, c;
    for(n = 0x00; n <= 0x0F; n++)
        if(n > 4 && n < 11)
            c = 0x01000000 * (n - 5) + 0x02020507;
        else
            c = 0x07020504;
        colorset[n] = c;

    for(n = 0x10; n <= 0x1F; n++)
        colorset[n] = 0x10100000 * (n & 0xF) + 0x0B0A0507;

    for(n = 0x20; n <= 0x2F; n++)
        colorset[n] = 0x10100000 * (n & 0xF) + 0x0D0C0507;

    for(n = 0x30; n <= 0x3F; n++)
        colorset[n] = 0x10100000 * (n & 0xF) + 0x080E0507;
}


/*
 * print a new line
 */
static void newline(void)
{
    uint16_t* sp = (uint16_t*)tvPtr->screen;
    col = 0;
    row &= ~1;  // make sure rows is even
    row += 2;
    if (row == TV_TEXT_ROWS) {
        row-=2;
        longmove((uint32_t*)sp, (uint32_t*)&sp[TV_TEXT_COLS*2], TV_TEXT_LASTROW>>1); // scroll
        wordfill(&sp[TV_TEXT_LASTROW], blank, TV_TEXT_COLS*2); // clear new line
    }
}

/*
 * print a character
 */
static void printc(int c)
{
    int n1 = row * TV_TEXT_COLS + col;
    int n2 = (row+1) * TV_TEXT_COLS + col;
    uint32_t val = 0;
    
    val  = ((gcolor << 1) | (c & 1)) << 10;
    val += 0x200 + (c & 0xFE);

    // Driver updates during invisible. Need some delay so screen updates right.
    // For flicker-free once per scan update, you can wait for status != invisible.
    // while(*tvPtr->status != TV_TEXT_STAT_INVISIBLE)    ;

    tvPtr->screen[n1] = val;
    tvPtr->screen[n2] = val | 1;

    if (++col == TV_TEXT_COLS)
        newline();
}

/*
 * TV_Text setTileColor sets tile data color at x,y position
 * See header file for more details.
 */
uint16_t tvText_getTileColor(int x, int y)
{
    int shift  = 11;
    int   mask = ((TV_TEXT_COLORS-1) << shift);
    int   ndx  = y * TV_TEXT_COLS + x;
    int   color = 0;
    
    if(x >= TV_TEXT_COLS)
        return 0; 
    if(y >= TV_TEXT_ROWS)
        return 0;
    color = tvPtr->screen[ndx] & mask;
    color >>= shift;
    return color;
}


/*
 * TV_Text setTileColor sets tile data color at x,y position
 * See header file for more details.
 */
void tvText_setTileColor(int x, int y, uint16_t color)
{
    uint16_t tile = 0;
    int shift  = 11;
    int   mask = ((TV_TEXT_COLORS-1) << shift);
    int   ndx  = y * TV_TEXT_COLS + x;
    
    while(tvPtr->status != TV_TEXT_STAT_INVISIBLE)
        ;
    if(x >= TV_TEXT_COLS)
        return; 
    if(y >= TV_TEXT_ROWS)
        return;

    color <<= shift; 
    tile = tvPtr->screen[ndx];
    tile = tile & ~mask;
    tile = tile | color;
    tvPtr->screen[ndx] = tile;
}

/*
 * TV_Text str function prints a string at current position
 * See header file for more details.
 */
void    tvText_str(char* sptr)
{
    while(*sptr) {
#ifdef TV_TEXT_OUT
        if(*sptr == 0xA)    // translate to newline
            tvText_out(0xD);
        else
            tvText_out(*(sptr));
#else
        putchar(*(sptr));
#endif
        sptr++;
    }
}

/*
 * TV_Text dec function prints a decimal number at current position
 * See header file for more details.
 */
void    tvText_dec(int value)
{
    int n = value;
    int len = 10;
    int result = 0;

    if(value < 0) {
        value = ~value;
        printc('-');
    }

    n = 1000000000;

    while(--len > -1) {
        if(value >= n) {
            printc(value / n + '0');
            value %= n;
            result++;
        }
        else if(result || n == 1) {
            printc('0');
        }
        n /= 10;
    }
}

/*
 * TV_Text hex function prints a hexadecimal number at current position
 * See header file for more details.
 */
void    tvText_hex(int value, int digits)
{
    int ndx;
    char hexlookup[] =
    {
        '0', '1', '2', '3', '4', '5', '6', '7',
        '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
    };
    while(digits-- > 0) {
        ndx = (value >> (digits<<2)) & 0xf;
        printc(hexlookup[ndx]);
    }   
}


/*
 * TV_Text bin function prints a binary number at current position
 * See header file for more details.
 */
void    tvText_bin(int value, int digits)
{
    int bit = 0;
    while(digits-- > 0) {
        bit = (value >> digits) & 1;
        printc(bit + '0');
    }   
}

#define TV_TEXT_OUT
#ifdef TV_TEXT_OUT
/*
 * TV_Text out function prints a character at current position or performs
 * a screen function.
 * See header file for more details.
 */
void    tvText_out(int c)
{
    if(flag == 0)
    {
        switch(c)
        {
            case 0:
                wordfill(tvPtr->screen, gcolor << 11 | blank, TV_TEXT_SCREENSIZE);
                col = 0;
                row = 0;
                break;
            case 1:
                col = 0;
                row = 0;
                break;
            case 8:
                if (col)
                    col--;
                break;
            case 9:
                do {
                    printc(' ');
                } while(col & 7);
                break;
            case 0xA:   // fall though
            case 0xB:   // fall though
            case 0xC:   // fall though
                flag = c;
                return;
            case 0xD:
                newline();
                break;
            default:
                printc(c);
                break;
        }
    }
    else
    if (flag == 0xA) {
        col = c % TV_TEXT_COLS;
    }
    else
    if (flag == 0xB) {
        row = (c<<1) % TV_TEXT_ROWS;
    }
    else
    if (flag == 0xC) {
        gcolor = c & 0xf;
    }
    flag = 0;
}
#else

/*
 * TV_Text print null terminated char* to screen with normal stdio definitions
 * See header file for more details.
 */
void    print(char* s)
{
    while(*s) {
        putchar(*(s++));
    }
}

/*
 * TV_Text putchar print char to screen with normal stdio definitions
 * See header file for more details.
 */
int     putchar(char c)
{
    switch(c)
    {
        case '\b':
            if (col)
                col--;
            break;
        case '\t':
            do {
                printc(' ');
            } while(col & 7);
            break;
        case '\n':
            newline();
            break;
        case '\r':
            col = 0;
            break;
        default:
            printc(c);
            break;
    }
    return (int)c;
}
#endif

/*
 * TV_Text getTile gets tile data from x,y position
 * See header file for more details.
 */
uint16_t   tvText_getTile(int x, int y)
{
    if(x >= TV_TEXT_COLS)
        return 0;
    if(y >= TV_TEXT_ROWS)
        return 0;
    return tvPtr->screen[y * TV_TEXT_COLS + x];
}

/*
 * TV_Text setTile sets tile data at x,y position
 * See header file for more details.
 */
void tvText_setTile(int x, int y, uint16_t tile)
{
    if(x >= TV_TEXT_COLS)
        return;
    if(y >= TV_TEXT_ROWS)
        return;
    tvPtr->screen[y * TV_TEXT_COLS + x] = tile;
}

/*
 * TV_Text setCurPosition function sets position to x,y.
 * See header file for more details.
 */
void    tvText_setCurPosition(int x, int y)
{
    col = x;
    row = y<<1;
}

/*
 * TV_Text setCoordPosition function sets position to Cartesian x,y.
 * See header file for more details.
 */
void    tvText_setCoordPosition(int x, int y)
{
    col = x;
    row = TV_TEXT_ROWS-(y<<1)-1;
}

/*
 * TV_Text setXY function sets position to x,y.
 * See header file for more details.
 */
void    tvText_setXY(int x, int y)
{
    col = x;
    row = y<<1;
}

/*
 * TV_Text setX function sets column position value
 * See header file for more details.
 */
void    tvText_setX(int value)
{
    col = value;
}

/*
 * TV_Text setY function sets row position value
 * See header file for more details.
 */
void    tvText_setY(int value)
{
    row = value<<1;
}

/*
 * TV_Text setYhalf function sets half row position value
 * See header file for more details.
 */
void    tvText_setYhalf(int value)
{
    row = value;
}

/*
 * TV_Text getX function gets column position
 * See header file for more details.
 */
int tvText_getX(void)
{
    return col;
}

/*
 * TV_Text getY function gets row position
 * See header file for more details.
 */
int tvText_getY(void)
{
    return row;
}

/*
 * TV_Text setColors function sets palette color set index
 * See header file for more details.
 */
void tvText_setColors(int value)
{
    gcolor = value % TV_TEXT_COLORS;
}

/*
 * TV_Text getColors function gets palette color set index
 * See header file for more details.
 */
int tvText_getColors(void)
{
    return gcolor % TV_TEXT_COLORS;
}

/*
 * TV_Text getWidth function gets screen width.
 * See header file for more details.
 */
int tvText_getColumns(void)
{
    return TV_TEXT_COLS;
}

/*
 * TV_Text getHeight function gets screen height.
 * See header file for more details.
 */
int tvText_getRows(void)
{
    return TV_TEXT_ROWS;
}

/*
 * Draw a text button
 */
void tvText_button(int left, int top, int width, int height, int color, char* text)
{
  int n = 0;
  int len = strlen(text);

  if(width < len)
    width = len;
  tvText_box(left,top,width,height,color);
  row = top+1;
  col = left+1;
  for(n=0; n < width; n++)
    tvText_out(*text++);
  if(width > len)
    for(n=len; n < width; n++)
       tvText_out(' ');
}

int boxindex = 0;
int boxcolor = 0;

static void boxchr(int c)
{
    int val;
    //i := (color << 1 + c & 1) << 10 + 0x200 + c & 0xFE
    val = (boxcolor << 10) + 0x200 + c;
    //i := ((boxcolor << 10) + 0x200 + c) | 1'& 0xFE
    tvPtr->screen[boxindex++] = val;
}

/*
 * Draw a box
 */
void tvText_box(int left, int top, int width, int height, int color)
{
    int n;
    color <<= 1;
    boxcolor = color | 1;
    boxindex = top * TV_TEXT_COLS + left;
    boxchr(0x0);
    for(n = width-1; n > -1; n--)
        boxchr(0xc);
    boxchr(0x8);
    for(n = 1; n <= height; n++)
    {
        boxindex = (top + n) * TV_TEXT_COLS + left;
        boxchr(0xA);
        boxindex += width;
        boxchr(0xB);
    }
    boxindex = (top + height + 1) * TV_TEXT_COLS + left;
    boxchr(0x1);
    for(n = width-1; n > -1; n--)
        boxchr(0xc+1);
    boxchr(0x9);
}


/*
 * Draw a filled rectangle
 */
void tvText_rectangle(int left, int top, int width, int height, int color, int fill)
{
    int x,y,n,c;

    if(fill)
    {
        c = 0x20;
        color = c & 0xF;
        for(y=0; y < height; y++) {
            for(x = 0; x < width; x++) {
                n = (((color << 1) + (c & 1)) << 10) + 0x200 + (c & 0xFE);
                tvPtr->screen[(y+top ) * TV_TEXT_COLS + x+left ] = n;
            }
        }
    }
    else
    {
        c = 0x20;
        color = c & 0xF;
        y = 0;
        for(x = 0; x < width; x++) {
            n = (((color << 1) + (c & 1)) << 10) + 0x200 + (c & 0xFE);
            tvPtr->screen[(y+top ) * TV_TEXT_COLS + x+left ] = n;
        }
        y = height;
        for(x = 0; x < width; x++) {
            n = (((color << 1) + (c & 1)) << 10) + 0x200 + (c & 0xFE);
            tvPtr->screen[(y+top ) * TV_TEXT_COLS + x+left ] = n;
        }
        x = 0;
        for(y = 0; y < height; y++) {
            n = (((color << 1) + (c & 1)) << 10) + 0x200 + (c & 0xFE);
            tvPtr->screen[(y+top ) * TV_TEXT_COLS + x+left ] = n;
        }
        x = width;
        for(y = 0; y < height; y++) {
            n = (((color << 1) + (c & 1)) << 10) + 0x200 + (c & 0xFE);
            tvPtr->screen[(y+top ) * TV_TEXT_COLS + x+left ] = n;
        }
    }
}

/*
+------------------------------------------------------------------------------------------------------------------------------+
                                                   TERMS OF USE: MIT License                                                                                                              
+------------------------------------------------------------------------------------------------------------------------------
Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation     
files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
is furnished to do so, subject to the following conditions:                                                                   
                                                                                                                              
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
                                                                                                                              
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
+------------------------------------------------------------------------------------------------------------------------------+
*/
