/**
 * @file FdSerial.c
 * Full Duplex Serial adapter module.
 *
 * Copyright (c) 2008, Steve Denson
 * See end of file for terms of use.
 */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <propeller.h>

#include "FdSerial.h"

static long  gFdSerialCog = 0;  //cog flag/id
static long  gclkfreq;          // clock frequency

static FdSerial_t *gpFdSerial;  // fdserial descriptor pointer

/*
 * delay function used after start cog and during screen updates
 * @param per is number of ticks to wait ....
 */
static void wait(int per)
{
    // not the best way to do it !
    volatile int time = 0;
    for(time = 0; time < per; )
        time++;
}

/**
 * start initializes and starts native assembly driver in a cog.
 * @param rxpin is pin number for receive input
 * @param txpin is pin number for transmit output
 * @param mode is interface mode. see header FDSERIAL_MODE_...
 * @param baudrate is frequency of bits ... 115200, 57600, etc...
 * @returns non-zero on success
 */
int FdSerial_start(int rxpin, int txpin, int mode, int baudrate)
{
    extern int _binary_FullDuplexSerial_dat_start;
    extern int _binary_FullDuplexSerial_dat_end;

    int     error   = 0;
    void    *pasm   = 0;
    uint8_t *bufp   = 0;

    int binsize = &_binary_FullDuplexSerial_dat_end-&_binary_FullDuplexSerial_dat_start;
    iprintf("binary size %d bytes\n", binsize);

    gpFdSerial  = (FdSerial_t*) _syscall(&error,SYS_huballoc,sizeof(FdSerial_t));
    pasm        = (uint32_t*) _syscall(&error,SYS_huballoc,binsize*4);
    longmove(pasm,&_binary_FullDuplexSerial_dat_start,binsize);

    // get HUB buffer space. rx/tx buffers must be contiguous.
    // split 2x buffer in two for rx/tx later.
    bufp        = (uint8_t*) _syscall(&error,SYS_huballoc,(FDSERIAL_BUFF_SIZE<<1));

    FdSerial_stop();
    gpFdSerial->rx_pin = rxpin;             // recieve pin
    gpFdSerial->tx_pin = txpin;             // transmit pin
    gpFdSerial->mode   = mode;              // interface mode
    gclkfreq = CLKFREQ;                     // hack to read clock freq from HUB addr 0
    gpFdSerial->ticks = gclkfreq/baudrate;  // baud
    gpFdSerial->rxbuff = (uint32_t) bufp;

    iprintf("clkfrq %08x\n", gclkfreq);
    iprintf("pasm   %08x\n", (uint32_t)pasm);
    iprintf("fdPtr  %08x\n", (uint32_t)gpFdSerial);

    iprintf("rx_pin %08x %08x\n", &gpFdSerial->rx_pin, gpFdSerial->rx_pin);
    iprintf("tx_pin %08x %08x\n", &gpFdSerial->tx_pin, gpFdSerial->tx_pin);
    iprintf("mode   %08x %08x\n", &gpFdSerial->mode  , gpFdSerial->mode  );
    iprintf("ticks  %08x %08x\n", &gpFdSerial->ticks , gpFdSerial->ticks );
    iprintf("rxhead %08x %08x\n", &gpFdSerial->rx_head,gpFdSerial->rx_head);
    iprintf("rxtail %08x %08x\n", &gpFdSerial->rx_tail,gpFdSerial->rx_tail);
    iprintf("txhead %08x %08x\n", &gpFdSerial->tx_head,gpFdSerial->tx_head);
    iprintf("txtail %08x %08x\n", &gpFdSerial->tx_tail,gpFdSerial->tx_tail);
    iprintf("rxbuf  %08x %08x\n", &gpFdSerial->rxbuff, gpFdSerial->rxbuff);

    iprintf("\nNow Stop Console before FdSerial start\n");
    _syscall(&error,SYS_stopuart);

    gFdSerialCog = cognew(pasm, (void*)gpFdSerial) + 1;
    wait(1000); // give cog chance to load

    return gFdSerialCog;
}

/**
 * stop stops the cog running the native assembly driver 
 */
void FdSerial_stop(void)
{
    int id = gFdSerialCog - 1;
    if(gFdSerialCog > 0) {
        cogstop(id);
    }
}
/**
 * rxflush empties the receive queue 
 */
void FdSerial_rxflush(void)
{
    while(FdSerial_rxcheck() >= 0)
        ; // clear out queue by receiving all available 
}

/**
 * Gets a byte from the receive queue if available
 * Function does not block. We move rxtail after getting char.
 * @returns receive byte 0 to 0xff or -1 if none available 
 */
int FdSerial_rxcheck(void)
{
    int rc = -1;
    FdSerial_t* p = gpFdSerial;
    char* rxbuff = gpFdSerial->rxbuff;
    if(p->rx_tail != p->rx_head)
    {
        rc = rxbuff[p->rx_tail];
        p->rx_tail = (p->rx_tail+1) & FDSERIAL_BUFF_MASK;
    }
    return rc;
}

/**
 * Get a byte from the receive queue if available within timeout period.
 * Function blocks if no recieve for ms timeout.
 * @param ms is number of milliseconds to wait for a char
 * @returns receive byte 0 to 0xff or -1 if none available 
 */
int FdSerial_rxtime(int ms)
{
    int rc = -1;
#ifdef TODO
    int t0 = 0;
    int t1 = 0;
    asm("mov %t0, cnt");
    do {
        rc = FdSerial_rxcheck();
        asm("mov %t1, cnt");
        if((t1 - t0)/(gclkfreq/1000) > ms)
            break;
    } while(rc < 0);
#endif
    return rc;
}

/**
 * Wait for a byte from the receive queue. blocks until something is ready.
 * @returns received byte 
 */
int FdSerial_rx(void)
{
    int rc = FdSerial_rxcheck();
    while(rc < 0)
        rc = FdSerial_rxcheck();
    return rc;
}

/**
 * tx sends a byte on the transmit queue.
 * @param txbyte is byte to send. 
 */
int FdSerial_tx(int txbyte)
{
    int rc = -1;
    FdSerial_t* p = gpFdSerial;

    char* txbuff = gpFdSerial->rxbuff + FDSERIAL_BUFF_SIZE; // tx buffer is contiguous

    while(p->tx_tail != p->tx_head) // wait for queue to be empty
        ;

    //while(p->tx_tail == ((p->tx_head+1) & FDSERIAL_BUFF_MASK)) ; // wait for queue to be empty
    txbuff[p->tx_head] = txbyte;
    p->tx_head = (p->tx_head+1) & FDSERIAL_BUFF_MASK;
    if(p->mode & FDSERIAL_MODE_IGNORE_TX_ECHO)
        rc = FdSerial_rx(); // why not rxcheck or timeout ... this blocks for char
    return rc;
}

/**
 * out sends a byte on the transmit queue.
 * @param txbyte is byte to send. 
 */
int FdSerial_out(int txbyte)
{
    return FdSerial_tx(txbyte);
}

#ifndef FDS_DISABLE_OUTS
/**
 * tx sends a string on the transmit queue.
 * @param sp is the null terminated string to send. 
 */
void FdSerial_str(char* sp)
{
    while(*sp)
        FdSerial_tx(*(sp++));
}

/**
 * dec prints a string representation of a decimal number to output
 * @param value is number to print. 
 */
void FdSerial_dec(int value)
{
    int n = value;
    int len = 10;
    int result = 0;

    if(value < 0) {
        value = ~value;
        FdSerial_tx('-');
    }

    n = 1000000000;

    while(--len > -1) {
        if(value >= n) {
            FdSerial_tx(value / n + '0');
            value %= n;
            result++;
        }
        else if(result || n == 1) {
            FdSerial_tx('0');
        }
        n /= 10;
    }
}

/**
 * hex prints a string representation of a hexadecimal number to output
 * @param value is number to print. 
 * @param digits is number of characters to print. 
 */
void FdSerial_hex(int value, int digits)
{
    int ndx;
    char hexlookup[] =
    {
        '0', '1', '2', '3', '4', '5', '6', '7',
        '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
    };
    while(digits-- > 0) {
        ndx = (value >> (digits<<2)) & 0xf;
        FdSerial_tx(hexlookup[ndx]);
    }
}

/**
 * bin prints a string representation of a binary number to output
 * @param value is number to print. 
 * @param digits is number of characters to print. 
 */
void FdSerial_bin(int value, int digits)
{
    int bit = 0;
    while(digits-- > 0) {
        bit = (value >> digits) & 1;
        FdSerial_tx(bit + '0');
    }
}
#endif
// FDS_DISABLE_OUTS

/*
+------------------------------------------------------------------------------------------------------------------------------+
                                                   TERMS OF USE: MIT License                                                                                                              
+------------------------------------------------------------------------------------------------------------------------------
Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation     
files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
is furnished to do so, subject to the following conditions:                                                                   
                                                                                                                              
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
                                                                                                                              
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
+------------------------------------------------------------------------------------------------------------------------------+
*/
