'' +--------------------------------------------------------------------------+
'' | Cluso's TriBladeProp:  Blade #2 Driver                            v0.201 |
'' +--------------------------------------------------------------------------+
'' |  Author:            "Cluso99" (Ray Rodrick)                              |
'' |  Copyright (c) 2009 "Cluso99" (Ray Rodrick)                              |
'' |  License            MIT License - See end of file for terms of use       |
'' +--------------------------------------------------------------------------+
                             
'' RR20090320   First version -v006 (uses SRAM) "sdram_rr006.spin"
''              Interface looks like "sdspiqasm.spin"
''              maximum SRAM disk is 512KB + (512KB - 64KB) = 960KB
'' RR20090321   General purpose SRAM driver for Blade #2 "SRAM_driver_011.spin"
''              change sector size to 256 and shift 8
''              _012 fix err:=true/false
'' RR20090323   _013 simulate 137 byte sectors in 128 bytes ($A7 $0E $D2 ...=128=... $B7 $C8 $CD $A5 $2B $00)
''              _015 do not clear on start (bug anyway not clearing)
''              _016 add C=clear and M=move (still w/r still not returning correct data - corrupted by z80 ram???
'' RR20090330   _020 modify for contiguous 1MB SRAM generic
'' RR20090330   "TriBladeProp_Blade2_200.spin" Hardware driver for Blade #2
''              _201 fix parameter order
'' RR20090429   _202 add idle/active modes to giveup/reclaim the bus to/from the SD driver


'' NOTE: THIS FIXES THE SD CONFLICT BUT THERE IS STILL POSSIBLE CONFLICT WITH THE ZICOG DIRECT RAM ACCESS ???



'' +--------------------------------------------------------------------------+
'' | Cluso's TriBladeProp:  Hardware interface definitions                    |
'' +--------------------------------------------------------------------------+
'' NOTE: 1. This driver should be used to drive all hardware on Blade #2.
''          Specifically, this cog drives the LE pin (P27) so that nothing should use
''          this pin once the code is initialised. (i.e. EEPROM access, since SDA = LE pin)
''       2. Before calling this routine, set DIRA=%xx000000_00000000_00000000_00000000 to avoid conflict.
''       3. P0...P29 are used by this routine, but P30 & P31 are not used.
''       4. When this routine completes, P27 will be left enabled (DIRA) and set to =0 (OUTA).
''          The latch will be left with -CE(U23)=0 (i.e. SRAM U23 enabled)
''          and A19=0, A20=0. All other latched pins will be =1.
''       5. If you stop the driver, LE will float high and the latched value will be destroyed.
''       6. Currently NO testing is done for ram address + length => 512KB (CAUSES BUS CONFLICT)
''
'' This driver will be extended to provide microSD, Flash and Eeprom routines.
''
'       P0-7   = D0-7  
'       P8-15  = A0-7    Latches A19,A20,-CE(U23),-CE(U24),-RST1,-RST3,-CS(uSD),-CS(U26)
'       P15-23 = A8-15 
'       P24-26 = A16-18
'       P27    = SDA      LE (U25 Latch on low) - driven by driver cog while running
'       P28    = SCL     -OE (U24+U24 SRAM)
'       P29    = WE      -WE (U23+U24 SRAM)
'       P30    = SO      + not used by this driver
'       P31    = SI      +
'
'                                          +-------- -CS (U26 FLASH)   -+
'                                          |+------- -CS (J22 microSD)  +- Latched pins...
'                                          ||+------ -RST3              |    passes when LE=1
'             (P30) SO ---+                |||+----- -RST1              |    latches on  LE low going edge
'             (P31) SI --+|                ||||+---- -CE (U24 SRAM-hi)  |    latched on  LE=0
'                        ||+---- -WE(SDA)  |||||+--- -CE (U23 SRAM-lo)  |
'                        |||+--- -OE(SCL)  ||||||+-- A20                |
'                        ||||+--  LE       |||||||+- A19               -+
'                        |||||             ||||||||
'          P31...P0 --> %00xxx_xxxxxxxxxxx_xxxxxxxx_xxxxxxxx  
'                              +---------+ +------+ +------+
'                              A18......A8 A7....A0 D7....D0
'                                          Q7....Q0
'
'
VAR

' communication params(5) between cog driver code - only "command" and "errx" are modified by the driver
   long  command, hubaddrs, ramaddrs, blocklen, errx, cog ' rendezvous between spin and assembly (can be used cog to cog)
'        command  = R, W, C, M (read/write/clear/move a block of sram) =0 when operation completed by cog
'        hubaddrs = hub address for data buffer
'        ramaddrs = ram address for data ($0_0000...$7_FFFF for U23 512KB, $1_0000...$F_FFFF for U24 512KB = 1MB total)
'        blocklen = ram buffer length for data transfer
'        errx     = returns =0 (false=good), else <>0 (true & error code)
'        cog      = cog no of driver (set by spin start routine)
   
PUB start : err_
' Initialise the TriBladeProp Blade #2 driver (and latches SRAM(U23))
  command := "I"
  cog := 1 + cognew(@tbp2_start, @command)
  if cog == 0
    err_ := $FF                 ' error = no cog
  else
    repeat while command        ' driver cog sets =0 when done
    err_ := errx                ' driver cog sets =0 if no error, else xx = error code

PUB stop
   if cog
      cogstop(cog~ - 1)      

PUB idle : err_
' Place the bus in the idle state (for SD driver use)
  command := "i"                ' idle command
' Wait for command to complete and get status
  repeat while command          ' driver cog sets =0 when done
  err_ := errx                  ' driver cog sets =0 if no error, else xx = error code

PUB active : err_
' Activate the bus (from SD driver use)
  command := "a"                ' active command
' Wait for command to complete and get status
  repeat while command          ' driver cog sets =0 when done
  err_ := errx                  ' driver cog sets =0 if no error, else xx = error code


PUB DoCmd(command_, hub_address, ram_address, block_length) : err_
' Do the command: R, W, C, M (read/write/clear/move a block of sram)
  hubaddrs := hub_address       ' hub address start
  ramaddrs := ram_address       ' ram address start
  blocklen := block_length      ' block length
  command  := command_          ' must be last !!
' Wait for command to complete and get status
  repeat while command          ' driver cog sets =0 when done
  err_ := errx                  ' driver cog sets =0 if no error, else xx = error code


DAT
'' +--------------------------------------------------------------------------+
'' | Cluso's TriBladeProp:  Blade #2 Cog Driver                               |
'' +--------------------------------------------------------------------------+
                        org     0
tbp2_start    ' setup the pointers to the hub command interface (saves execution time later
                                      '  +-- These instructions are overwritten as variables after start
comptr        mov     comptr, par     ' -|  hub pointer to command                
hubptr        mov     hubptr, par     '  |  hub pointer to hub address            
ramptr        add     hubptr, #4      '  |  hub pointer to ram address            
lenptr        mov     ramptr, par     '  |  hub pointer to length                 
errptr        add     ramptr, #8      '  |  hub pointer to error status           
cmd           mov     lenptr, par     '  |  command  I/R/W/G/P/Q                  
hubaddr       add     lenptr, #12     '  |  hub address                           
ramaddr       mov     errptr, par     '  |  ram address                           
len           add     errptr, #16     '  |  length                                
err           nop                     ' -+  error status returned (=0=false=good) 


' Initialise hardware (latches -CE(U23)=0 A19=0 A20=0 and makes LE=0 and output)
init                    mov     err, #0                 ' reset err=false=good
                        mov     outa, ram_enable_lo     ' put latch back to SRAM=U23
                        mov     dira, ram_dir_read      ' enable bits (for init & wrblock)
                        xor     outa, ram_LE_bit        ' LE= 1 -> 0 (latch)
                        mov     dira, ram_LE_bit        ' only LE is output
'                       mov     outa, ram_latched       ' not really required
' set done & status
done                    wrlong  err, errptr             ' status  =0=false=good, else error x
                        wrlong  zero, comptr            ' command =0 (done)
' wait for a command (pause short time to reduce power)
pause                   mov     ctr, delay      wz      ' if =0 no pause
              if_nz     add     ctr, cnt
              if_nz     waitcnt ctr, #0                 ' wait for a short time (reduces power)
                        rdlong  cmd, comptr     wz      ' command ?
              if_z      jmp     #pause                  ' not yet
' decode command
                        cmp     cmd, #"R"       wz      ' R = read block
              if_z      jmp     #rdblock
                        cmp     cmd, #"W"       wz      ' W = write block
              if_z      jmp     #wrblock
'                       cmp     cmd, #"C"       wz      ' C = clear
'             if_z      jmp     #???
'                       cmp     cmd, #"M"       wz      ' M = move (a ram block)
'             if_z      jmp     #???
                        cmp     cmd, #"i"       wz      ' i = idle (give up) the bus to the SD card
              if_z      jmp     #tristate
                        cmp     cmd, #"a"       wz      ' a = active (reclaim) the bus from the SD card
              if_z      jmp     #activate
                        mov     err, cmd                ' error = cmd (unknown command)
                        jmp     #done

'Give up the bus
tristate                mov     dira, #0                ' all inputs
                        mov     outa, #0                ' and =0
                        jmp     #done

'Activate (reclaim) the bus
activate                jmp     #init                   'reset outputs and latch

'---------------------------------------------------------------------------------------------------------
'Memory Access Functions

'Read a "len" byte block given by "ramaddr" into hub at "hubaddr"
rdblock                 call    #ram_open               ' prepare for reading block
                                                        ' returns with outa = ramaddr = -WE=1 -OE=1 LE=0 + address<<8
                        xor     outa, ram_OE_bit        ' -OE = 1 -> 0
                        nop                             ' sram delay (should not be reqd)
'rdloop                 wrbyte  ina, bufadr             ' <===== illegal - cannot use ina in wrxxxx instruction
rdloop                  mov     outx, ina               ' read byte from SRAM \ ignores upper bits
                        wrbyte  outx, hubaddr           ' copy byte to hub    /
                        add     hubaddr, #1             ' inc hub pointer
                        add     outa, #(1 << 8)         ' inc sram address
                        djnz    len, #rdloop            ' loop for xxx bytes
                        jmp     #init

'Write a "len" byte block given by "ramaddr" from hub at "hubaddr"
wrblock                 call    #ram_open               ' prepare for reading block
                                                        ' returns with outa = ramaddr = -WE=1 -OE=1 LE=0 + address<<8
                        mov     dira, ram_dir_write     ' set D0-7 outputs also
wrloop                  rdbyte  outx, hubaddr           ' copy byte from hub
                        or      outx, ramaddr           ' add -WE=1 -OE=1 LE=0 + address<<8 to D0-7 bits
                        mov     outa, outx              ' -WE=1 -OE=1 LE=0 + address + data
                        xor     outa, ram_WE_bit        ' -WE= 1 -> 0  \\ write pulse
                        nop                             ' sram delay   ||
                        xor     outa, ram_WE_bit        ' -WE= 0 -> 1  //
                        add     hubaddr, #1             ' inc hub pointer
                        add     ramaddr, #(1 << 8)      ' inc sram address
                        djnz    len, #wrloop            ' loop for xxx bytes
'                       mov     dira, ram_dir_read      ' put D0-7 back to inputs (not really required)
                        jmp     #init

'SRAM access setup
ram_open                rdlong  hubaddr, hubptr         ' get hub address
                        rdlong  ramaddr, ramptr         ' get ram address
                        rdlong  len, lenptr             ' get length
                        mov     err, #5                 ' err=5
                        mov     outx, ram_enable_lo     ' -WE=1 -OE=1 LE=1, -CE(U23 SRAM) = 0 (A19-20=0)
                        cmp     ramaddr, ram_top1 wc,wz ' < 512KB?
              if_b      jmp     #ram_open5
                        sub     ramaddr, ram_top1       ' - 512KB
                        mov     outx, ram_enable_hi     ' -WE=1 -OE=1 LE=1, -CE(U24 SRAM) = 0 (A19-20=0)
                        cmp     ramaddr, ram_top1 wc,wz ' < 512KB? (i.e. < 1MB?)
              if_ae     jmp     #done                   ' error "5"
ram_open5               mov     outa, outx              ' -WE=1 -OE=1 LE=1, -CE(U23)=0 or -CE(U24)= 0 (A19-20=0)
                        mov     dira, ram_dir_read      ' set outputs
                        xor     outa, ram_LE_bit        ' LE= 1 -> 0 (latch)
                        shl     ramaddr, #8             ' move address to bits P8..P26
                        or      ramaddr, ram_latched    ' -WE=1 -OE=1 LE=0 + address<<8
                        mov     outa, ramaddr           ' setup read address incl -WE=1 -OE=1 LE=0
ram_open_ret            ret



' TriBladeProp Blade #2 Hardware Access:
'
'                                          +-------- -CS (U26 FLASH)   -+                                    
'                                          |+------- -CS (J22 microSD)  +- Latched pins...                   
'                                          ||+------ -RST3              |    passes when LE=1                
'             (P30) SO ---+                |||+----- -RST1              |    latches on  LE low going edge   
'             (P31) SI --+|                ||||+---- -CE (U24 SRAM-hi)  |    latched on  LE=0                
'                        ||+---- -WE(SDA)  |||||+--- -CE (U23 SRAM-lo)  |                                    
'                        |||+--- -OE(SCL)  ||||||+-- A20                |                                    
'                        ||||+--  LE       |||||||+- A19               -+                                    
'                        |||||             ||||||||                                                          
ram_WE_bit      long    %00100_00000000000_00000000_00000000 ' WE bit
ram_OE_bit      long    %00010_00000000000_00000000_00000000 ' OE bit
ram_LE_bit      long    %00001_00000000000_00000000_00000000 ' LE bit
ram_latched     long    %00110_00000000000_00000000_00000000 ' -WE=1 -OE=1 LE=0                                 
ram_enable_lo   long    %00111_00000000000_11111000_00000000 ' -WE=1 -OE=1 LE=1 + -CE(U23 SRAM) = 0 (A19-20=0)
ram_enable_hi   long    %00111_00000000000_11110100_00000000 ' -WE=1 -OE=1 LE=1 + -CE(U24 SRAM) = 0 (A19-20=0)
ram_dir_read    long    %00111_11111111111_11111111_00000000 ' outputs = WE, OE, LE, A0-18, inputs D0-7
ram_dir_write   long    %00111_11111111111_11111111_11111111 ' outputs = WE, OE, LE, A0-18, D0-7
'                              +---------+ +------+ +------+
'                              A18......A8 A7....A0 D7....D0
'                                          Q7....Q0

delay         long      80             ' waitcnt delay to reduce power (#80 = 1uS approx)
ctr           long      0              ' used to pause execution (lower power use) & byte counter
zero          long      0              ' constant 0
outx          long      0              ' as a temp address + WE/OE/LE bits
ram_top1      long      (512 * 1024)   ' 512KB (one sram)


{{
+------------------------------------------------------------------------------------------------------------------------------+
|                                                   TERMS OF USE: MIT License                                                  |                                                            
+------------------------------------------------------------------------------------------------------------------------------+
|Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    | 
|files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    |
|modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software|
|is furnished to do so, subject to the following conditions:                                                                   |
|                                                                                                                              |
|The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.|
|                                                                                                                              |
|THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          |
|WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         |
|COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   |
|ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         |
+------------------------------------------------------------------------------------------------------------------------------+
}} 