CON

' *****************************************************************************
' *                                                                           *
' *     AiChip Industries LCC Virtual Machine                   Kernel.Spin   *
' *                                                                           *
' *****************************************************************************

' <<KERNEL FOR LCC>>

CON

' <<INSERT HEADER>>
' <<FINISH HEADER>>

' <<IF DEBUG>>
' <<OR TEST>>
' <<OR SPIN>>
' <<OR OBJECT>>
' <<ELSE>>
' <<SET SPIN>>
' <<END IF>>

' <<LIST DEBUG    Target   : Debuggable Spin File>>
' <<LIST TEST     Target   : Testable Spin File>>
' <<LIST SPIN     Target   : Standalone Spin File>>
' <<LIST OBJECT   Target   : Embeddable Object>>

' <<LIST SOURCE   Source   : >>
' <<LIST DATE     Created  : >>

' <<LIST COMPILE  Compiler : >>
' <<LIST MAKESPIN MakeSpin : >>
' <<LIST KERNEL   Kernel   : >>

' <<LIST OPTIONS  Options  : >>

' <<IF DEBUG>>
' <<OR TEST>>
' <<OR SPIN>>

CON

' *****************************************************************************
' *                                                                           *
' *     Configuration                                                         *
' *                                                                           *
' *****************************************************************************

CON

' .---------------------------------------------------------------------------.
' |     Define Board Configuration                                            |
' `---------------------------------------------------------------------------'

' <<INSERT CLKMODE>>

  _CLKMODE      = XTAL1+PLL16x
  _XINFREQ      = 7_372_800                     ' 118MHz Operation !

' <<FINISH CLKMODE>>

' <<END IF>>

' <<IF DEBUG>>
' <<OR TEST>>

CON

' .---------------------------------------------------------------------------.
' |     Define Hardware Configuration                                         |
' `---------------------------------------------------------------------------'

' <<INSERT TVPIN>>

  TV_PIN        = 12

' <<FINISH TVPIN>>

' <<IF DEBUG>>

  ' << IF KERNEL.SPIN>>
  RX_PIN        = 31
  TX_PIN        = 30
  XX_BAUDRATE   = 9600
  XX_INVERT     = false
  ' <<END IF>>

' <<INSERT SXPIN>>

  SI_PIN        = 8
  SO_PIN        = 9
  SX_BAUDRATE   = 9600
  SX_INVERT     = false

' <<FINISH SXPIN>>

' <<END IF>>

' <<END IF>>

' <<IF DEBUG>>

' <<DEFINE BEBUGMODE>>

CON

' .---------------------------------------------------------------------------.
' |     Define Debugging Modes                                                |
' `---------------------------------------------------------------------------'

  DEBUG_STACK   = false
  DEBUG_SP_INFO = false
  DEBUG_FRAME   = false
  DEBUG_SP_FULL = false

  DEBUG_STEP_MS = 500

' <<FINISH DEBUGMODE>>

OBJ

' .---------------------------------------------------------------------------.
' |     Uses AiChip_CogDebug for debugging the Cog                            |
' `---------------------------------------------------------------------------'

' <<INSERT COGDEBUG>>

  debug         : "AiChip_CogDebug_008"

' <<FINISH COGDEBUG>>

' <<END IF>>

' <<IF DEBUG>>
' <<OR TEST>>

OBJ

' .---------------------------------------------------------------------------.
' |     Uses TV Display For debug reports                                     |
' `---------------------------------------------------------------------------'

  tv            : "TV_Text"

' <<END IF>>

' <<IF DEBUG>>
' <<OR TEST>>

CON

' *****************************************************************************
' *                                                                           *
' *     Main Program                                                          *
' *                                                                           *
' *****************************************************************************

' <<END IF>>

' <<IF KERNEL.SPIN>>

CON

' .---------------------------------------------------------------------------.
' |     Timing Test                                                           |
' `---------------------------------------------------------------------------'

PUB Kernel_TimingTest | varPtr

  ' Pasm :  28s  3,571,428 increments per second        1:1
  ' Spin : 640s    156,250 increments per second       23:1
  ' Lcc  : 510s    196,078 increments per second       18:1   byte reads
  ' Lcc  : 410s    243,900 increments per second       15:1   optimised reads
  ' Lcc  : 390s    256,410 increments per second       14:1   with branches
  ' Lcc  : 370s    270,270 increments per second       13:1   with branches to get var

  kernel_debug ' @@@@@@@@@@@

  tv.Start( TV_PIN )

  tv.Str( Kernel_Version )
  if Kernel_SourceFileName
    tv.Str( String( " - " ) )
    tv.Str( Kernel_SourceFileName )
  tv.Out( $0D )

  long[ varPtr := Kernel_GetVarAddress(0) ] := 0

  tv.Str( String( "Watching : " ) )
  tv.Hex( varPtr,4 )
  tv.Out( ":" )
  tv.Hex( varPtr-@LccCode,4 )
  case Kernel_GetVarSize(0)
    1 : tv.Str( String( " BYTE " ) )
    2 : tv.Str( String( " WORD " ) )
    3 : tv.Str( String( " CHAR " ) )
    4 : tv.Str( String( " LONG " ) )
    5 : tv.Str( String( " REAL " ) )
  tv.Str( Kernel_GetVarName(0) )
  tv.Out( $0D )
  tv.Out( $0D )

  Kernel_Start

  Debug.RunForever

  repeat while long[varPtr] < 10_000_000
    tv.Out( $0A )
    tv.Out(   0 )
    tv.Dec( long[varPtr] )

  tv.Str( String( " - FINISHED" ) )

' <<END IF>>

' <<SELECT CASE>>

' <<CASE DEBUG>>

CON

' .---------------------------------------------------------------------------.
' |     Debugging                                                             |
' `---------------------------------------------------------------------------'

{
PRI CheckBase      | objectTable, methodCount, objectCount

  tv.hex(debug.getbaseaddr,8)
  tv.out($0D)

  objectTable := $0010
  methodCount := byte[objectTable+2]+1
  objectCount := byte[objectTable+3]
  repeat until ( objectTable + objectCount*4 + methodCount*4 ) == debug.getbaseaddr
    objectTable := word[objectTable+0]
    methodCount := byte[objectTable+2]+1
    objectCount := byte[objectTable+3]
  tv.hex(objectTable,8)
  tv.out("<")
  tv.out($0D)

  objectTable := $0010
  methodCount := byte[$0012]+1
  objectCount := byte[$0013]
  objectTable += methodCount*4
  tv.hex(objectTable,8)
  tv.out($0D)
  repeat objectCount
    tv.Hex(word[objectTable],4)
    tv.out($0D)
    objectTable +=4

  repeat
  repeat
}

PUB Kernel_Debug | xSp, xPc, xOp, xArgPtr, xLocPtr, spCount

  tv.Start( TV_PIN )

{
  checkbase
}

  if SO_PIN => 0
    debug.StartSerial( SI_PIN, SO_PIN, SX_BAUDRATE, SX_INVERT )
    if SO_PIN == 30
      tv.Str( String( "Paused for serial debugger start ...",$0D ) )
      PauseMs( 2500 )
  ' <<IF KERNEL.SPIN>>
  else
    debug.StartSerial( RX_PIN, TX_PIN, XX_BAUDRATE, XX_INVERT )
    tv.Str( String( "Paused for serial debugger start ...",$0D ) )
    PauseMs( 2500 )
  ' <<END IF>>

  Kernel_Start

  debug.Out($0D)
  Str( Kernel_Version )
  if Kernel_SourceFileName
    Str( String( " - " ) )
    Str( Kernel_SourceFileName )
  Str( String( $0D,"Debugging ..." ) )
  debug.Out( $0D)

  repeat
    Debug.RunTo( Kernel_Label(@Fetch)+1 )
    tv.Out( $0D )

    if DEBUG_FRAME
      Out( $0D )

    ' Display stack details
    if DEBUG_STACK
      xSp := Debug.Peek( Kernel_Label(@sp) )
      xLocPtr :=  Debug.Peek( Kernel_Label(@locPtr) )
      spCount := ( (xLocPtr-xSp) ~> 2 )+1
      ' Show sp / locPtr addresses
      if DEBUG_SP_INFO
        Hex( xSp,8)
        Out( " " )
        Hex( xLocPtr, 8)
        Out( " " )
        Dec( spCount)
        Out( $0D)
      else
        Dec( spCount )
        Out( " ")
      if spCount > 0
        repeat spCount <# 3
          Hex( long[ xSp ],8 )
          Out( " " )
          xSp += 4
          spCount--
        if spCount > 0
          Out( "+" )
        Out($0D)

    ' Display stack frame
    if DEBUG_FRAME
      xSp := Debug.Peek( Kernel_Label(@sp) )
      xArgPtr :=  Debug.Peek( Kernel_Label(@argPtr) )
      xLocPtr :=  Debug.Peek( Kernel_Label(@locPtr) )
      if xLocPtr > xSp
        ShowFrame( String( "ARG" ), xArgPtr , xLocPtr )
        ShowFrame( String( "LOC" ), xLocPtr , xSp     )
        ShowFrame( String( "SP " ), xSp     , xSp     )
      else
        ShowFrame( String( "ARG" ), xArgPtr , xSp     )
        ShowFrame( String( "SP " ), xSp     , xLocPtr )
        ShowFrame( String( "LOC" ), xLocPtr , xLocPtr )

    ' Display Top of stack
    if DEBUG_SP_FULL
      xSp := Debug.Peek( Kernel_Label(@sp) )
      spCount := 7
      debug.Hex( xSp, 4 )
      repeat while ( spCount-- ) and ( xSp < $6000 )
        debug.Out( " " )
        debug.Hex( long[ xSp ] , 8 )
        xSp +=4
      debug.Out( $0D )

    ' Disassemble LCC bytecode to be executed
    xPc := Debug.Peek( Kernel_Label(@pc) )
    xSp := Debug.Peek( Kernel_Label(@sp) )
    xOp := Debug.Peek( Kernel_Label(@OVERLAY_opc) )
    Hex( xSp, 4 )
    Out( " " )
    debug.Hex( xPc, 4 )
    debug.Out( ":" )
    Hex( xPc-@LccCode, 4 )
    Out( " " )
    Hex( xOp, 2 )
    Out( " " )
    Str( Kernel_FixOpcName( xPc, xOp, Kernel_GetOpcName(xOp) ) )
    debug.Out($0D)

    ' Delay between steps
    PauseMs( DEBUG_STEP_MS )

PRI PauseMs( mS )
  if mS
    WaitCnt( CLKFREQ / 1000 * mS + CNT )

PRI ShowFrame( strPtr, fromAdr, toAdr )
  Str( strPtr )
  Out( " " )
  Hex( fromAdr, 4 )
  Out( " " )
  Hex( long[ fromAdr ] , 8 )
  Out( $0D )
  repeat while ( fromAdr -= 4 ) > toAdr
    Str( String( "    " ) )
    Hex( fromAdr, 4 )
    Out( " " )
    Hex( long[ fromAdr ] , 8 )
    Out( $0D )

PRI Kernel_Label( atCogAdr )
  if atCogAdr => @OVERLAY_END
    return word[ atCogAdr ]
  return Debug.Label( atCogAdr )

PRI Kernel_FixOpcName( pcVal, opcVal, namPtr ) | i, n, digits
  digits := 4
  result := namPtr
  case opcVal & $F0
    $80   : opcVal &= 3
            if ( opcVal & $0C ) == $08 ' #ARG+
              opcVal <<= 2
            n := 0
            digits := 2 << opcVal
            repeat
              n := ( n << 8 ) | byte[++pcVal]
            while opcVal--
    $90   : n := ( ( opcVal & $0F ) << 28 ) ~> 28       ' #MEM+/#MEM-
            n := pcVal - @LccCode + n + 1
    $A0   : n := ( opcVal & $0F ) << 2                  ' #ARG+
    $B0   : n := opcVal & $0F                           ' #LOC+
    other :
      if opcVal < $80
        n := |< ( opcVal & $1F )
        if opcVal & $20
          n--
        if opcVal & $40
          n ^= $FFFF_FFFF
        digits := 8
      else
        return
  i := namPtr + StrSize(namPtr) - 8
  repeat 8
    byte[ i++ ] := Kernel_HexDigit( n >> 28 )
    n <<= 4
  byte[ i ] := 0

' <<CASE TEST>>

CON

' .---------------------------------------------------------------------------.
' |     Program Test                                                          |
' `---------------------------------------------------------------------------'

PUB Kernel_Testing | varCount, varNum, varPtr, n

  tv.Start( TV_PIN )

  tv.Str( Kernel_Version )
  if Kernel_SourceFileName
    tv.Str( String(" - ") )
    tv.Str( Kernel_SourceFileName )
  tv.Out( $0D )
  tv.Out( $0D )

  ' <<IF KERNEL.VARS>>

  varCount := Maximum( Kernel_GetVarCount, 9 )

  if varCount
    repeat varNum from 0 To varCount-1
      varPtr := Kernel_GetVarAddress(varNum)
      tv.Hex( varPtr,4 )
      tv.Out( ":" )
      tv.Hex( varPtr-@LccCode,4 )
      case Kernel_GetVarSize(varNum)
        1 : tv.Str( String( "  BYTE  " ) )
        2 : tv.Str( String( "  WORD  " ) )
        2 : tv.Str( String( "  CHAR  " ) )
        4 : tv.Str( String( "  LONG  " ) )
        5 : tv.Str( String( "  REAL  " ) )
      tv.Str( Kernel_GetVarName(varNum) )
      tv.Out( $0A )
      tv.Out(  30 )
      tv.Str( String( "       ", $0D ) )
  else
    tv.Str( String( "No variables" ) )

  ' <<END IF>>

  Kernel_Start

' <<IF KERNEL.VARS>>

  repeat
    repeat varNum from 0 To varCount-1
      tv.Out( $0B )
      tv.Out( varNum+2 )
      tv.Out( $0A )
      varPtr := Kernel_GetVarAddress(varNum)
      case Kernel_GetVarSize(varNum)
        1 : tv.Out( 37 )
            tv.Hex( Kernel_PeekByte(varPtr), 2 )
        2 : tv.Out( 35 )
            tv.Hex( Kernel_PeekWord(varPtr), 4 )
        4 : tv.Out( 31 )
            tv.Hex( Kernel_PeekLong(varPtr), 8 )
    ' @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
    tv.out($0D)
    repeat varNum from $7200 to $721C step 4
      tv.hex(word[varNum],4)
      tv.out(" ")
    tv.out($0D)
    repeat varNum from $7300 to $731C step 4
      tv.hex(word[varNum],4)
      tv.out(" ")
    ' @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

' <<ELSE>>
  tv.Str( String( "No variables" ) )
' <<END IF>>

' <<IF KERNEL.VARS>>

PRI Maximum( a, b )
  return a <# b
  
PRI Kernel_PeekByte( varPtr )
  return byte[varPtr]

PRI Kernel_PeekWord( varPtr )
  return byte[varPtr] | ( byte[varPtr+1] << 8 )

PRI Kernel_PeekLong( varPtr )
  return Kernel_PeekWord(varPtr) | ( Kernel_PeekWord(varPtr+2) << 16 )

' <<END IF>>

' <<END SELECT>>

' <<IF DEBUG>>
' <<OR TEST>>

' <<IF KERNEL.SPIN>>

CON

' *****************************************************************************
' *                                                                           *
' *     Support Utilities                                                     *
' *                                                                           *
' *****************************************************************************

' <<END IF>>

' <<IF DEBUG>>

CON

' .---------------------------------------------------------------------------.
' |     Opcode Information                                                    |
' `---------------------------------------------------------------------------'

PRI Kernel_GetOpcCount : opcCount | opcPtr

  repeat while byte[opcPtr]
    opcCount++
    opcPtr++
    repeat while byte[opcPtr++]

PRI Kernel_GetOpcName( opcVal ) : namPtr

  if opcVal< $C0
    case opcVal & $F0
      $80   : if ( opcVal& $C0 )== 0
                namPtr := String( "NUM.K?  #$????????" )
              else
                namPtr := String( "NUM.K?? #???+$????????" )
                case opcVal & $0C
                  $4 : bytemove( namPtr+6, String("M #MEM"), 6 )
                  $8 : bytemove( namPtr+6, String("A #ARG"), 6 )
                  $C : bytemove( namPtr+6, String("L #LOC"), 6 )
              byte[namPtr+5] := ( opcVal & $03 ) + "1"
      $90   : namPtr := String( "NUM.MZ? #MEM+$????????" )
              byte[namPtr+6] := Kernel_HexDigit( opcVal & $0F )
      $A0   : namPtr := String( "NUM.AZ? #ARG+$????????" )
              byte[namPtr+6] := Kernel_HexDigit( opcVal & $0F )
      $B0   : namPtr := String( "NUM.LZ? #LOC+$????????" )
              byte[namPtr+6] := Kernel_HexDigit( opcVal & $0F )
      other : if opcVal
                namPtr := String( "NUM.KP? #$????????" )
                byte[namPtr+6] := ( ( opcVal & $60 ) >> 4 ) + "0"
              else
                namPtr := String( "BRK" )
  else
    namPtr := @OpcNames
    repeat while ( byte[namPtr] <> 0 ) and ( opcVal <> byte[namPtr] )
      namPtr++
      repeat while byte[namPtr++]
    namPtr++

PRI Kernel_HexDigit( n )
  if ( result := ( n & $F )+ "0" ) > "9"
    result += constant( "A"-"9"-1 )

' <<END IF>>

' <<IF KERNEL.VARS>>

CON

' .---------------------------------------------------------------------------.
' |        Variable Information                                               |
' `---------------------------------------------------------------------------'

PRI Kernel_GetVarCount : varCount | varPtr

  varPtr := @VarNames
  repeat while byte[varPtr]
    varCount++
    varPtr += 3
    repeat while byte[varPtr++]

PRI Kernel_GetVarName( varNum ) : varPtr

  varPtr := @VarNames
  repeat while varNum--
    varPtr += 3
    repeat while byte[varPtr++]
  varPtr += 3

PRI Kernel_GetVarAddress( varNum ) : varPtr

  varPtr := Kernel_GetVarName(varNum)
  varPtr := ( ( byte[varPtr-2] << 8 ) | byte[varPtr-1] ) + @LccCode

PRI Kernel_GetVarSize( varNum ) : varSize

  varSize := byte[Kernel_GetVarName(varNum)-3]

' <<END IF>>
' <<END IF>>

' <<IF OBJECT>>

CON

' *****************************************************************************
' *                                                                           *
' *     Exported Interfaces                                                   *
' *                                                                           *
' *****************************************************************************

PUB SetStack( argStackBase, argStackSize )

  Kernel_SetStack( argStackBase, argStackSize )

' <<INSERT METHODS>>

PUB Main

  Kernel_Start

PUB FunctionWithArgs( _1, _2 ) ' This is just an example to show the style

  Kernel_Push( _1 )
  Kernel_Push( _2 )
  Kernel_Run( @LccCode + $0000 )

' <<FINISH METHODS>>

' <<END IF>>

' <<IF DEBUG>>
' <<OR TEST>>
' <<OR OBJECT>>

CON

' *****************************************************************************
' *                                                                           *
' *     Version Information                                                   *
' *                                                                           *
' *****************************************************************************

PUB Kernel_Version

  ' <<INSERT KERNELVERSION>>

  return String( "AiChip LccVm 002" )

  ' <<FINISH KERNEL VERSION>>

PUB Kernel_SourceFileName

  ' <<INSERT SOURCEFILE>>

  return String( "Kernel.c" )

  ' <<FINISH SOURCEFILE>>

' <<END IF>>

' <<IF DEBUG>>

CON

' *****************************************************************************
' *                                                                           *
' *     Display Interface for debugging                                       *
' *                                                                           *
' *****************************************************************************

PRI Str( strPtr )

  if TV_PIN => 0
    tv.Str(strPtr)
  if SO_PIN => 0
    debug.Str(strPtr)

PRI Out( n )

  if TV_PIN => 0
    tv.Out(n)
  if SO_PIN => 0
    debug.Out(n)

PRI Hex( n , w )

  if TV_PIN => 0
    tv.Hex(n,w)
  if SO_PIN => 0
    debug.Hex(n,w)

PRI Bin( n , w )

  if TV_PIN => 0
    tv.Bin(n,w)
  if SO_PIN => 0
    debug.Bin(n,w)

PRI Dec( n )

  if TV_PIN => 0
    tv.Dec(n)
  if SO_PIN => 0
    debug.Dec(n)

' <<END IF>>

CON

' *****************************************************************************
' *                                                                           *
' *     Kernel Launcher                                                       *
' *                                                                           *
' *****************************************************************************

VAR

  long  paramPc
  long  paramArgPtr
  long  paramSp
  long  paramLocPtr
  ' <<IF KERNEL.LIBS>>
  long  paramLccBase
  ' <<END IF>>

' <<IF SPIN>>

PUB Kernel_AutoRun

  paramPc      := @LccCode

  paramSp      := $8000   - 4
  paramArgPtr  := paramSp - 4
  paramLocPtr  := paramSp - 4

  ' <<IF KERNEL.LIBS>>
  paramLccBase := @LccVm
  ' <<END IF>>

  CogNew( @LccVm, @paramPc )

' <<ELSE>>

PUB Kernel_Start

  Kernel_Run( @LccCode )

PUB Kernel_Call( startPc )

  Kernel_Run( startPc )

PUB Kernel_Run( startPc )

  paramPc := startPc

  ' <<IF DEBUG>>
  ' <<OR TEST>>
  Kernel_SetStack(0,$6000)
  ' <<ELSE>>
  Kernel_SetStack(0,$8000)
  ' <<END IF>>

  Kernel_Push(0)
  long[ paramSp ] := 0

  ' <<IF KERNEL.LIBS>>
  paramLccBase := @LccVm
  ' <<END IF>>

  ' <<IF DEBUG>>
  Debug.CogNew_WithDebugger( @LccVm, @paramPc, @DEBUG_BLOCK )

  repeat while paramPc <> 0

    ' <<IF KERNEL.SPIN>>
    return
    ' <<END IF>>
  ' <<ELSE>>
  CogNew( @LccVm, @paramPc )

  repeat while paramPc <> 0
  ' <<END IF>>

PUB Kernel_SetStack( argStackBase, argStackSize )

  if paramSp == 0
    paramSp     := argStackBase + argStackSize - 4
    paramArgPtr := paramSp
    paramLocPtr := paramSp

PUB Kernel_Push( argParameter )

  ' <<IF DEBUG>>
  ' <<OR TEST>>
  Kernel_SetStack(0,$6000)
  ' <<ELSE>>
  Kernel_SetStack(0,$8000)
  ' <<END IF>>

  long[ paramSp ] := argParameter

  paramSp     -= 4
  paramLocPtr -= 4

' <<END IF>>

' <<IF DEBUG>>

CON

' *****************************************************************************
' *                                                                           *
' *     Debug Support                                                         *
' *                                                                           *
' *****************************************************************************

DAT

DEBUG_BLOCK   long      0[3]

' <<END IF>>

CON

' *****************************************************************************
' *                                                                           *
' *     LCC Virtual Machine                                                   *
' *                                                                           *
' *****************************************************************************

  '                Opcode ZCW
  PASM_MOV      = %101000_001
  PASM_DJNZ     = %111001_001
  PASM_TJNZ     = %111010_000
  PASM_TJZ      = %111011_000

DAT

' *****************************************************************************
' *                                                                           *
' *     Initialisation                                                        *
' *                                                                           *
' *****************************************************************************

'       If no arguments provided ...    If arguments provided ...
'
'                       .--------.                      .--------.
'       argPtr -------> |  "pc"  |      argPtr -------> | ARG+0  |
'                       |--------|                      | ARG+4  |
'       sp -------.---> |  zero  |                      | ARG+8  |
'       locPtr ---'     `--------'                      |--------|
'                                                       |  "pc"  |
'                                                       |--------|
'                                       sp -------.---> |  zero  |
'                                       locPtr ---'     `--------'
'
'       The "pc" is always zero - This is so that when the EXT executes it will
'       pull a zero address from the stack frame and we know we can halt and
'       return control to the caller. The caller can monitor "pc" and it will
'       be set to non-zero when it finished.

LccVm           org     $000

pc              call    #GetArg
argPtr          call    #GetArg
sp              call    #GetArg
locPtr          call    #GetArg

' <<IF KERNEL.SPIN>>
                jmp     #hubOffset
' <<END IF>>
' <<IFNOT KERNEL.LIBS>>
                mov     k_LccCode,pc
' <<ELSE>>
hubOffset       call    #GetArg
lmmPc           sub     hubOffset,k_LccVm
      IF_ALWAYS add     k_LibPtrTable,hubOffset
      IF_ALWAYS add     k_LccCode,hubOffset
' <<END IF>>

                wrlong  argNxt,PAR                      ' Clear paramPc as we're running

tmp             jmp     #Fetch

DAT

' <<IF KERNEL.KPX>>

' *****************************************************************************
' *                                                                           *
' *     Packed Number Handling ( $00 .. $7F )                                 *
' *                                                                           *
' *         .---------  Bit Number Indicator    n := 1 << b                   *
' *         |                                                                 *
' *     %0cdb bbbb                                                            *
' *       ||                                                                  *
' *       |`---------- Decrement Bit            n := n-1                      *
' *       |                                                                   *
' *       `----------- Inversion Bit            n := n ^ $FFFF_FFFF           *
' *                                                                           *
' *     Note that the Decrement is applied before the Inversion when both     *
' *     bits are set. Note that to get $0000_0001 the opcode used is not      *
' *     $00 but $21. This is forced by the compilation process and is done    *
' *     to keep $00 free as a special purpose opcode should it be needed.     *
' *                                                                           *
' *     The packing algorithm is slighty different to the one used in the     *
' *     Parallax Spin Interpreter bytecode but is the same principle.         *
' *                                                                           *
' *****************************************************************************
' *                                                                           *
' *     0cdbbbbb        000xxxxx      001xxxxx      010xxxxx      011xxxxx    *
' *                                                                           *
' *     xxx00000        00000001      00000000      FFFFFFFE      FFFFFFFF    *
' *     xxx00001        00000002      00000001      FFFFFFFD      FFFFFFFE    *
' *     xxx00010        00000004      00000003      FFFFFFFB      FFFFFFFC    *
' *     xxx00011        00000008      00000007      FFFFFFF7      FFFFFFF8    *
' *     xxx00100        00000010      0000000F      FFFFFFEF      FFFFFFF0    *
' *     xxx00101        00000020      0000001F      FFFFFFDF      FFFFFFE0    *
' *     xxx00110        00000040      0000003F      FFFFFFBF      FFFFFFC0    *
' *     xxx00111        00000080      0000007F      FFFFFF7F      FFFFFF80    *
' *     xxx01000        00000100      000000FF      FFFFFEFF      FFFFFF00    *
' *     xxx01001        00000200      000001FF      FFFFFDFF      FFFFFE00    *
' *     xxx01010        00000400      000003FF      FFFFFBFF      FFFFFC00    *
' *     xxx01011        00000800      000007FF      FFFFF7FF      FFFFF800    *
' *     xxx01100        00001000      00000FFF      FFFFEFFF      FFFFF000    *
' *     xxx01101        00002000      00001FFF      FFFFDFFF      FFFFE000    *
' *     xxx01110        00004000      00003FFF      FFFFBFFF      FFFFC000    *
' *     xxx01111        00008000      00007FFF      FFFF7FFF      FFFF8000    *
' *     xxx10000        00010000      0000FFFF      FFFEFFFF      FFFF0000    *
' *     xxx10001        00020000      0001FFFF      FFFDFFFF      FFFE0000    *
' *     xxx10010        00040000      0003FFFF      FFFBFFFF      FFFC0000    *
' *     xxx10011        00080000      0007FFFF      FFF7FFFF      FFF80000    *
' *     xxx10100        00100000      000FFFFF      FFEFFFFF      FFF00000    *
' *     xxx10101        00200000      001FFFFF      FFDFFFFF      FFE00000    *
' *     xxx10110        00400000      003FFFFF      FFBFFFFF      FFC00000    *
' *     xxx10111        00800000      007FFFFF      FF7FFFFF      FF800000    *
' *     xxx11000        01000000      00FFFFFF      FEFFFFFF      FF000000    *
' *     xxx11001        02000000      01FFFFFF      FDFFFFFF      FE000000    *
' *     xxx11010        04000000      03FFFFFF      FBFFFFFF      FC000000    *
' *     xxx11011        08000000      07FFFFFF      F7FFFFFF      F8000000    *
' *     xxx11100        10000000      0FFFFFFF      EFFFFFFF      F0000000    *
' *     xxx11101        20000000      1FFFFFFF      DFFFFFFF      E0000000    *
' *     xxx11110        40000000      3FFFFFFF      BFFFFFFF      C0000000    *
' *     xxx11111        80000000      7FFFFFFF      7FFFFFFF      80000000    *
' *                                                                           *
' *****************************************************************************

NUM_KP          mov     lhs,#1
                shl     lhs,opc
                test    opc,#$20        WC
        IF_C    sub     lhs,#1
                test    opc,#$40        WC
        IF_C    xor     lhs,k_FFFF_FFFF

' <<END IF>>

' *****************************************************************************
' *                                                                           *
' *     Push result to stack                                                  *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     Push a result onto the stack. We put it here so we have a fairly      |
' |     quick and efficient route for packed number handling which falls      |
' |     straight back into the Fetch loop when completed.                     |
' |                                                                           |
' |     Note that the stack grows down from the top of memory so we need      |
' |     to decrement the Stack Pointer to push. Also note that the Stack      |
' |     pointer always points to top of stack. This makes it easier and       |
' |     quicker to deal with unary opcodes. The Stack remains aligned on      |
' |     long boundaries all the time; only longs are ever pushed to the       |
' |     stack.                                                                |
' |                                                                           |
' `---------------------------------------------------------------------------'

Push            sub     sp,#4
                wrlong  lhs,sp

' *****************************************************************************
' *                                                                           *
' *     Main Fetch Loop                                                       *
' *                                                                           *
' *****************************************************************************

Fetch           rdbyte  opc,pc          WZ
                add     pc,#1

' <<IF KERNEL.KPX>>

                test    opc,#$80        WZ      ' Packed Number ( $00..$7F ) ?
        IF_Z    jmp     #NUM_KP                 ' Yes

' <<END IF>>

                cmps    opc,#$90        WZ, WC  ' Other Opcodes ( $90..$FF ) ?
        IF_AE   jmp     #Others                 ' Yes

' *****************************************************************************
' *                                                                           *
' *     Multi-Byte Number Handling ( $80 .. $8F )                             *
' *                                                                           *
' *             .-----  How many bytes follow   00 = 1 - Byte                 *
' *             |                               01 = 2 - Word                 *
' *             |                               10 = 3 - Triple               *
' *             |                               11 = 4 - Long                 *
' *     %1000 ttss                                                            *
' *           |                                                               *
' *           `------  Adjustment Offset        00 = None                     *
' *                                             01 = MEM+                     *
' *                                             10 = ARG+                     *
' *                                             11 = LOC+                     *
' *                                                                           *
' *****************************************************************************

NUM_KX          mov     lhs,#0
                mov     ptr,opc
                and     ptr,#3          WZ

                rdbyte  lhs,pc
                add     pc,#1

        IF_Z    jmp     #:Done

:Loop           rdbyte  tmp,pc
                add     pc,#1
                shl     lhs,#8
                or      lhs,tmp

                djnz    ptr,#:Loop

' .---------------------------------------------------------------------------.
' |     We have a number, now adjust it if it is an offset                    |
' `---------------------------------------------------------------------------'

:Done           test    opc,#4          WC      '  C
                test    opc,#8          WZ      ' Z

  IF_Z_AND_C    add     lhs,k_LccCode              ' 01 = %1000 01ss    #MEM+n

  IF_Z          jmp     #Push                   ' 00 = %1000 00ss    #n

AdjustArgLoc

  IF_NC         shl     lhs,#2                  '                    #ARG+n

                neg     lhs,lhs

  IF_NC         add     lhs,argPtr              ' 10 = %1000 10ss    #ARG+n
  IF_C          add     lhs,locPtr              ' 11 = %1000 11ss    #LOC+n

                jmp     #Push

' *****************************************************************************
' *                                                                           *
' *                                                                           *
' *                                                                           *
' *****************************************************************************

Others          cmps    opc,#$C0        WZ, WC  ' Normal bytecode ( $C0..$FF ) ?
        IF_AE   jmp     #Dispatch               ' Yes

' *****************************************************************************
' *                                                                           *
' *     %1001 nnnn      NUM.MZX #MEM+n     Fast Load Memory   ( $90 .. $9F )  *
' *                                                                           *
' *     %1010 nnnn      NUM.AZX #ARG+n     Fast Load Argument ( $A0 .. $AF )  *
' *                                                                           *
' *     %1011 nnnn      NUM.LZX #LOC+n     Fast Load Local    ( $B0 .. $BF )  *
' *                                                                           *
' *****************************************************************************

NUM_XZX         mov     lhs,opc
                and     lhs,#$F

                test    opc,#$10        WC      '  C
                test    opc,#$20        WZ      ' Z

        IF_NZ   jmp     #AdjustArgLoc           ' 10 - 1010 nnnn AZX
                                                ' 11 - 1011 nnnn LZX + AZX

                shl     lhs,#28                 ' 01 - 1001 nnnn MZX
                sar     lhs,#28
                add     lhs,pc
                jmp     #Push

' *****************************************************************************
' *                                                                           *
' *     SPR Addressing                                                        *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

SPR_I32         rdbyte  lhs,pc
                add     pc,#1
                or      lhs,k_8000_0000
                jmp     #Push

GetFromSpr      shr     ptr,#4          WC
        IF_NC   sub     ptr,#1
        IF_NC   jmp     #GetFromSpr_Ret

                or      ptr,#$1F0
                movs    GetfromSpr_Src,ptr
k_8000_0000     long    $8000_0000              ' NOP
GetFromSpr_Src  mov     lhs,0-0

                ' <<IF DEBUG>>
                and     ptr,#$1FF
                cmp     ptr,#CNT        WZ
        IF_Z    mov     lhs,#0
                ' <<END IF>>

GetFromSpr_Ret  ret

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PutToSpr        shr     ptr,#4          WC
        IF_NC   jmp     #PutToHub

                or      ptr,#$1F0
                movd    PutToSpr_Dst,ptr
k_4000_0000     long    $4000_0000              ' NOP
PutToSpr_Dst    mov     0-0,lhs
                jmp     #Fetch

PutToHub        sub     ptr,#1

PutToSpr_Ret    ret

' *****************************************************************************
' *                                                                           *
' *     Opcode Dispatch ( $C0 .. $FF )                                        *
' *                                                                           *
' *****************************************************************************

Dispatch        sub     opc,#$C0
                add     opc,#OpcJmpTable
  IF_ALWAYS     jmp     opc

' <<INSERT OPCJMPTABLE>>

OpcJmpTable     jmp     #JPI                    ' C0
                jmp     #ENT                    ' C1
                jmp     #INC_I32                ' C2

' <<FINISH OPCJMPTABLE>>

' *****************************************************************************
' *                                                                           *
' *     Stack Handling                                                        *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PopLhsAndRhsAndPtr      rdlong  ptr,sp
                        add     sp,#4

PopLhsAndRhs            rdlong  rhs,sp
                        add     sp,#4

PopLhs                  rdlong  lhs,sp
                        add     sp,#4

PopLhs_Ret
PopLhsAndRhs_Ret
PopLhsAndRhsAndPtr_Ret  ret

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PopLhsThenPtr           rdlong  lhs,sp
                        add     sp,#4

PopPtr                  rdlong  ptr,sp
                        add     sp,#4
PopPtr_Ret
PopLhsThenPtr_Ret       ret

' *****************************************************************************
' *                                                                           *
' *     Variable Handling                                                     *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PopPtrGetByte           call    #PopPtr

GetByte                 shl     ptr,#1  WC, NR
              IF_C      call    #GetFromSpr
              IF_C      jmp     #GetByte_Ret

                        rdbyte  lhs,ptr
GetByte_Ret
PopPtrGetByte_Ret       ret

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PopPtrGetWord           call    #PopPtr

GetWord                 shl     ptr,#1  WC, NR
              IF_C      call    #GetFromSpr
              IF_C      jmp     #GetWord_Ret

GetLongLsb              test    ptr,#1  WZ

              IF_Z      rdword  lhs,ptr
              IF_Z      jmp     #GetWord_Ret

                        rdbyte  lhs,ptr
                        add     ptr,#1
                        rdbyte  tmp,ptr
                        shl     tmp,#8
                        or      lhs,tmp
                        sub     ptr,#1
GetLongLsb_Ret
GetWord_Ret
PopPtrGetWord_Ret       ret

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

PopPtrGetLong           call    #PopPtr

GetLong                 shl     ptr,#1  WC, NR
              IF_C      call    #GetFromSpr
              IF_C      jmp     #GetLong_Ret

                        test    ptr,#3  WZ

              IF_Z      rdlong  lhs,ptr
              IF_Z      jmp     #GetLong_Ret

                        call    #GetLongLsb
                        add     ptr,#2       
                        rdbyte  tmp,ptr
                        add     ptr,#1
                        shl     tmp,#16
                        or      lhs,tmp
                        rdbyte  tmp,ptr
                        shl     tmp,#24
                        or      lhs,tmp
                        sub     ptr,#3
GetLong_Ret
PopPtrGetLong_Ret       ret

' *****************************************************************************
' *                                                                           *
' *     Subroutine and Function Calls                                         *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Mark calling argument prior to calling Subroutine or Function         |
' `---------------------------------------------------------------------------'

ARG_I32         mov     argNxt,sp
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Jump to Subroutine ( Void Function - No return Value )                |
' |---------------------------------------------------------------------------|
' |     Jump to Function   ( Result returned on Stack )                       |
' `---------------------------------------------------------------------------'

'       Before the call ...             After the call ...
'
'                       |--------|                      |--------|
'       argPtr -------> | ARG+0  |                      |        |
'                       | ARG+4  |                      |        |
'                       |--------|                      |--------|
'                       | XXXXX  |                      |        |
'                       | XXXXX  |                      |        |
'                       | XXXXX  |                      |        |
'                       | XXXXX  |                      |        |
'                       |--------|                      |--------|
'       locPtr -------> | LOC+0  |                      |        |
'                       | LOC+4  |                      |        |
'                       | LOC+8  |                      |        |
'                       |--------|                      |--------|
'       argNxt -------> | PAR+0  |      argPtr -------> | ARG+0  |
'                       | PAR+4  |                      | ARG+4  |
'                       |--------|                      |--------|
'                       | ?????  |                      | ?????  |
'                       |--------|                      |--------|
'       sp -----------> | dst pc |                      | argPtr |
'                       `--------'                      | locPtr |
'                                                       | pc     |
'                                                       |--------|
'                                       sp / locPtr --> |  zero  | For ENT
'                                                       `--------'

JFI_I32         or      argPtr,#1               ' Indicate we want a result

JSI             call    #PopPtr                 ' Pop where jump is to

                sub     sp,#4                   ' Push the current state
                wrlong  argPtr,sp
                sub     sp,#4
                wrlong  locPtr,sp
                sub     sp,#4
                wrlong  pc,sp

                mov     locPtr,sp               ' Set the new local base
                sub     locPtr,#4

                mov     argPtr,argNxt           ' Set the new argument base

                mov     pc,ptr                  ' Set the new pc

                mov     lhs,#0                  ' Push a zero local var count
                jmp     #Push

' .---------------------------------------------------------------------------.
' |     Enter Subroutine ( Allocate Local Variables )                         |
' `---------------------------------------------------------------------------'

'       On entry ( no push ) ...        After completion ...
'
'                       |--------|                      |--------|
'       argPtr -------> | ARG+0  |      argPtr -------> | ARG+0  |
'                       | ARG+4  |                      | ARG+4  |
'                       |--------|                      |--------|
'                       | ?????  |                      | ?????  |
'                       |--------|                      |--------|
'                       | XXXXXX |                      | XXXXXX |
'                       | XXXXXX |                      | XXXXXX |
'                       | XXXXXX |      sp ------. -N   | XXXXXX |
'                       |--------|               |      |--------|
'       sp / locPtr --> |  zero  |      locPtr --|----> | LOC+0  |
'                       `--------'               |      | LOC+4  |
'                                                |      | LOC+8  |
'                                                `----> | LOC+12 |
'                                                       `--------'

'       On entry ( with push ) ...      After completion ...
'
'                       |--------|                      |--------|
'       argPtr -------> | ARG+0  |      argPtr -------> | ARG+0  |
'                       | ARG+4  |                      | ARG+4  |
'                       | ARG+8  |                      | ARG+8  |
'                       |--------|                      |--------|
'                       | ?????  |                      | ?????  |
'                       |--------|                      |--------|
'                       | XXXXXX |                      | XXXXXX |
'                       | XXXXXX |                      | XXXXXX |
'                       | XXXXXX |      sp ------. -N   | XXXXXX |
'                       |--------|               |      |--------|
'       locPtr -------> |  zero  |      locPtr --|----> | LOC+0  |
'       sp -----------> | number |               |      | LOC+4  |
'                       `--------'               |      | LOC+8  |
'                                                `----> | LOC+12 |
'                                                       `--------'
'
'       We know locPtr is always 4 below the stack frame so we don't care if
'       there's been a number pushed or not, we simply pop whatever's on the
'       top ( the number or the zero ) and then reset sp := locPtr+4. Having
'       done that we can then go about adjusting the stack for the locals.
'
'       Note that the size of locals is in bytes and may not be a multiple of
'       four ( longs ). This is because locals do not have long alignment.
'       this is not a problem because our stack stays long aligned and any
'       access into the locals area is done as normal for other variables so
'       that handles non-aligned variables as it usually does.

ENT             call    #PopLhs                 ' Get the size of locals

                mov     sp,locPtr               ' Restore sp
                add     sp,#4

                sub     sp,lhs                  ' Create space for locals

:Loop           test    sp,#3           WZ      ' Make sure sp aligns to long
        IF_NZ   sub     sp,#1
        IF_NZ   jmp     #:Loop

                mov     argNxt,sp
                sub     argNxt,#4

                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Exit from Subroutine ( Void Function - No return Value )              |
' |---------------------------------------------------------------------------|
' |     Return from Function ( Leave Result on Stack )                        |
' `---------------------------------------------------------------------------'

'       Before the return ...           After the return ...
'
'                       |--------|                      |--------|
'                       | ARG+0  |      argPtr -------> | ARG+0  |
'                       | ARG+4  |                      | ARG+4  |
'                       |--------|                      |--------|
'                       | ?????  |                      | ?????  |
'                       |--------|                      |--------|
'                       | XXXXX  |                      | XXXXXX |
'                       | XXXXX  |                      | XXXXXX |
'                       | XXXXX  |                      | XXXXXX |
'                       |--------|                      |--------|
'                       | LOC+0  |      locPtr -------> | LOC+0  |
'                       | LOC+4  |                      | LOC+4  |
'                       | LOC+8  |                      | LOC+8  |
'                       |--------|                      |--------|
'                       | ?????  |                      | ?????  |
'                       | ?????  |  .-- sp -------.---> | ?????  |
'                       |--------|  |             |     |--------|
'                       | PutPtr |  |             |     | PutPtr |
'                       |--------|  |             |     |--------|
'       argPtr -------> | ARG+0  | -'             `---> | result |
'                       | ARG+4  |                      `--------'
'                       | ARG+8  |
'                       |--------|      PutPtr present if return from function
'                       | argPtr |
'                       | locPtr |
'                       | pc     |
'                       |--------|
'       locPtr -------> | LOC+0  |
'       sp -------.---> | LOC+4  |
'                 |     |--------|
'                 `---> | result |      Result present if return from function
'                       `--------'
'
'       Note that we pop pc first. This is so we can detect whether we have a
'       real stack frame or a faked one created when the code was started. If
'       it's fake then pc will be zero ...
'
'       If no arguments provided ...    If arguments provided ...
'
'                       .--------.                      .--------.
'                       |  zero  |                      | ARG+0  |
'                       |--------|                      | ARG+4  |
'       locPtr -------> | ?????? |                      | ARG+8  |
'                       `--------'                      |--------|
'                                                       |  zero  |
'                                                       |--------|
'                                       locPtr -------> | ?????? |
'                                                       `--------'
'
RET_I32         call    #PopLhs

EXT             mov     sp,locPtr
                add     sp,#4

                mov     tmp,argPtr

                rdlong  pc,sp           WZ      ' Restore PC

        IF_Z    jmp     #Halt                   ' Is zero so end of program

                add     sp,#4                   ' Restore locPtr
                rdlong  locPtr,sp

                add     sp,#4                   ' Restore argPtr
                rdlong  argPtr,sp

                add     sp,#4                   ' Get Put Ptr if function, rubbish if not
                rdlong  ptr,sp

                test    argPtr,#1       WC      ' Keep return value ?
                andn    argPtr,#1

                mov     sp,tmp                  ' Restore SP ( remove args )
                add     sp,#4

        IF_C    sub     sp,#4
        IF_C    wrlong  ptr,sp                  ' Put the Put Ptr back

        IF_C    jmp     #Push                   ' Yes
                jmp     #Fetch                  ' No

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

Halt
                jmp     #Halt

' *****************************************************************************
' *                                                                           *
' *     Conditional Execution                                                 *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Jump Unconditionally                                                  |
' `---------------------------------------------------------------------------'

JPI             Call    #PopPtr
                mov     pc,ptr
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Jump Boolean                                                          |
' `---------------------------------------------------------------------------'

JPT_I32         Call    #PopLhsAndRhs
                tjz     lhs,#Fetch
                mov     pc,rhs
                jmp     #Fetch

JPF_I32         Call    #PopLhsAndRhs
                tjnz    lhs,#Fetch
                mov     pc,rhs
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Jump Conditionally on Equality or Non-Equality                        |
' `---------------------------------------------------------------------------'

JEQ_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ
        IF_Z    mov     pc,ptr
                jmp     #Fetch

JNE_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ
        IF_NZ   mov     pc,ptr
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Jump Conditionally on Relative Values                                 |
' `---------------------------------------------------------------------------'

JGT_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ, WC
        IF_A    mov     pc,ptr
                jmp     #Fetch

JGE_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ, WC
        IF_AE   mov     pc,ptr
                jmp     #Fetch

JLT_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ, WC
        IF_B    mov     pc,ptr
                jmp     #Fetch

JLE_I32         Call    #PopLhsAndRhsAndPtr
                cmp     lhs,rhs         WZ, WC
        IF_BE   mov     pc,ptr
                jmp     #Fetch

JGT_S32         Call    #PopLhsAndRhsAndPtr
                cmps    lhs,rhs         WZ, WC
        IF_A    mov     pc,ptr
                jmp     #Fetch

JGE_S32         Call    #PopLhsAndRhsAndPtr
                cmps    lhs,rhs         WZ, WC
        IF_AE   mov     pc,ptr
                jmp     #Fetch

JLT_S32         Call    #PopLhsAndRhsAndPtr
                cmps    lhs,rhs         WZ, WC
        IF_B    mov     pc,ptr
                jmp     #Fetch

JLE_S32         Call    #PopLhsAndRhsAndPtr
                cmps    lhs,rhs         WZ, WC
        IF_BE   mov     pc,ptr
                jmp     #Fetch

' <<IF NORAM>>

' *****************************************************************************
' *                                                                           *
' *     ROM / Eeprom Handling                                                 *
' *                                                                           *
' *****************************************************************************

'       If we are running code from ROM we need a way to determine if an address
'       of #MEM+ refers to a global variable or to code. As there are no spare
'       bits in the opcodes the way we do this is by prefixing a #MEM+ to ROM
'       with a ROM opcode. We do this by setting a bit in k_LccCode above what it
'       would normally be so when it's added it can be vectored off elsewhere.
'       Once vectored we clear the bit so we're back pointing to variables.
'
ROM             or      k_LccCode,k_8000_0000
                jmp     #Fetch

' <<END IF>>

' *****************************************************************************
' *                                                                           *
' *     Variable Manipulation                                                 *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Get a Variable's contents whose Address is popped from the Stack      |
' `---------------------------------------------------------------------------'

GTI_I08         call    #PopPtrGetByte
                jmp     #Push

GTI_I16         call    #PopPtrGetWord
                jmp     #Push

GTI_I32         call    #PopPtrGetLong
                jmp     #Push

GTI_S08         call    #PopPtrGetByte
                shl     lhs,#24
                sar     lhs,#24
                jmp     #Push

GTI_S16         call    #PopPtrGetWord
                shl     lhs,#16
                sar     lhs,#16
                jmp     #Push

' .---------------------------------------------------------------------------.
' |     Set a byte variable's contents whose Address is popped from the Stack |
' `---------------------------------------------------------------------------'

DEC_I08         neg     rhs,#1
                jmp     #$+2
INC_I08         mov     rhs,#1
                call    #PopPtrGetByte
                call    #AddSkip1

PTI_I08         call    #PopLhsThenPtr

PutByte         shl     ptr,#1          WC, NR
      IF_C      call    #PutToSpr               ' May not return

                wrbyte  lhs,ptr
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Set a word variable's contents whose Address is popped from the Stack |
' `---------------------------------------------------------------------------'

DEC_I16         neg     rhs,#1
                jmp     #$+2
INC_I16         mov     rhs,#1
                call    #PopPtrGetWord
                call    #AddSkip1

PTI_I16         call    #PopLhsThenPtr

PutWord         shl     ptr,#1          WC, NR
        IF_C    call    #PutToSpr               ' May not return

PutLongMsb      test    ptr,#1          WZ, NR

        IF_Z    wrword  lhs,ptr
        IF_Z    jmp     #Fetch

                wrbyte  lhs,ptr
                add     ptr,#1
                shr     lhs,#8
                wrbyte  lhs,ptr
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |     Set a long variable's contents whose Address is popped from the Stack |
' `---------------------------------------------------------------------------'

DEC_I32         neg     rhs,#1
                jmp     #$+2
INC_I32         mov     rhs,#1
                call    #PopPtrGetLong
                call    #AddSkip1

PTI_I32         call    #PopLhsThenPtr

PutLong         shl     ptr,#1          WC, NR
      IF_C      call    #PutToSpr               ' May not return

                test    ptr,#3          WZ, NR

        IF_Z    wrlong  lhs,ptr
        IF_Z    jmp     #Fetch

                wrbyte  lhs,ptr
                add     ptr,#1
                shr     lhs,#8
                wrbyte  lhs,ptr
                add     ptr,#1
                shr     lhs,#8
                jmp     #PutLongMsb

                ' .-----------------------------------------------------------.
                ' |                                                           |
                ' `-----------------------------------------------------------'

AddSkip1        add     AddSkip1_Ret,#1
                add     lhs,rhs
AddSkip1_Ret    ret

' *****************************************************************************
' *                                                                           *
' *     Binary Operators                                                      *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Standard Maths                                                        |
' `---------------------------------------------------------------------------'

' <<LIB ADD 2>>

ADD_I32         call    #PopLhsAndRhs
                add     lhs,rhs
                jmp     #Push

' <<LIB SUB 2>>

SUB_I32         call    #PopLhsAndRhs
                sub     lhs,rhs
                jmp     #Push

' .---------------------------------------------------------------------------.
' |     Bit-Wise Operations                                                   |
' `---------------------------------------------------------------------------'

' <<LIB AND 2>>

AND_I32         call    #PopLhsAndRhs
                and     lhs,rhs
                jmp     #Push
' <<LIB OR 2>>

IOR_I32         call    #PopLhsAndRhs
                or      lhs,rhs
                jmp     #Push

' <<LIB XOR 2>>

XOR_I32         call    #PopLhsAndRhs
                xor     lhs,rhs
                jmp     #Push

' .---------------------------------------------------------------------------.
' |     Shift Operations                                                      |
' `---------------------------------------------------------------------------'

' <<LIB SHL2 1>>

SL2_I32         call    #PopLhs
                mov     rhs,#2
                jmp     #$+2

' <<LIB SHL 2>>

SHL_I32         call    #PopLhsAndRhs
                shl     lhs,rhs
                jmp     #Push

' <<LIB SHR2 1>>

SR2_I32         call    #PopLhs
                mov     rhs,#2
                jmp     #$+2

' <<LIB SHR 2>>

SHR_I32         call    #PopLhsAndRhs
                shr     lhs,rhs
                jmp     #Push

' <<LIB ROL 2>>

ROL_I32         call    #PopLhsAndRhs
                rol     lhs,rhs
                jmp     #Push

' <<LIB ROR 2>>

ROR_I32         call    #PopLhsAndRhs
                ror     lhs,rhs
                jmp     #Push

' <<LIB ASR 2>>

ASR_S32         call    #PopLhsAndRhs
                sar     lhs,rhs
                jmp     #Push

' *****************************************************************************
' *                                                                           *
' *     Unary Type Conversions                                                *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Shrink Unsigned Integer Size                                          |
' `---------------------------------------------------------------------------'

' <<LIB BYTE 1>>

I2I_I08         rdlong  lhs,sp
                and     lhs,#$FF
                jmp     #UnaryWriteBack

' <<LIB WORD 1>>

I2I_I16         rdlong  lhs,sp
                shl     lhs,#16
                shr     lhs,#16
                jmp     #UnaryWriteBack

' .---------------------------------------------------------------------------.
' |     Shrink Signed Integer Size                                            |
' `---------------------------------------------------------------------------'

S2S_S08         rdlong  lhs,sp
                shl     lhs,#24
                sar     lhs,#24
                jmp     #UnaryWriteBack

S2S_S16         rdlong  lhs,sp
                shl     lhs,#16
                sar     lhs,#16
                jmp     #UnaryWriteBack

' *****************************************************************************
' *                                                                           *
' *     Unary Operators                                                       *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Duplicate Top of Stack                                                |
' `---------------------------------------------------------------------------'

DUP_I32         rdlong  lhs,sp
                jmp     #Push

' .---------------------------------------------------------------------------.
' |     Numeric Negation                                                      |
' `---------------------------------------------------------------------------'

' <<LIB NEG 1>>

NEG_I32         rdlong  lhs,sp
                neg     lhs,lhs
                jmp     #UnaryWriteBack

' .---------------------------------------------------------------------------.
' |     Bit-Wise Inversion                                                    |
' `---------------------------------------------------------------------------'

' <<LIB NOT 1>>

NOT_I32         rdlong  lhs,sp
                xor     lhs,k_FFFF_FFFF

UnaryWriteBack  wrlong  lhs,sp
                jmp     #Fetch

' <<IF KERNEL.LIBS>>

DAT

' *****************************************************************************
' *                                                                           *
' *     LMM Interpreter                                                       *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Library call handler                                                  |
' `---------------------------------------------------------------------------'

LMM             call    #PopPtr                 ' Pop the LIB index

                shl     ptr,#2
                add     ptr,k_LibPtrTable

                rdlong  lmmPc,ptr               ' Get start address of code

                mov     rhs,lmmPc               ' Get the argument count
                and     rhs,#3          WZ

        IF_Z    jmp     #LMM_Jump               ' None

                mov     ptr,argPtr              ' Point to first argument

:Loop           rdlong  lhs,ptr                 ' Push arguments to stack
                sub     ptr,#4
                sub     sp,#4
                wrlong  lhs,sp
                djnz    rhs,#:Loop

' .---------------------------------------------------------------------------.
' |     The actual LMM interpreter                                            |
' `---------------------------------------------------------------------------'

Lmm_Jump        add     lmmPc,hubOffset

Lmm_Fetch       rdlong  Lmm_Opcode,lmmPc
                add     lmmPc,#4
Lmm_Opcode      mov     0-0,0-0
                jmp     #Lmm_Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "CALL #label" is coded as ...                                  |
' |                                                                           |
' |             jmp     #LMM_OP_CALL                                          |
' |             long    @label                                                |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_CALL     mov     0-0+LmmStack,lmmPc
                add     LMM_OP_CALL,k_0000_0200
                add     LMM_OP_RET,#1

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "JMP #label" is coded as ...                                   |
' |                                                                           |
' |             jmp     #LMM_OP_JMP                                           |
' |             long    @label                                                |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_JMP      rdlong  lmmPc,lmmPc
                jmp     #Lmm_Jump

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "RET" is coded as ...                                          |
' |                                                                           |
' |             jmp     #LMM_OP_RET                                           |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_RET      mov     lmmPc,0-0+LmmStack-1
                sub     LMM_OP_CALL,k_0000_0200
                sub     LMM_OP_RET,#1
                jmp     #Lmm_Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "MOV reg,k_NNNN_NNNN" is coded as ...                          |
' |                                                                           |
' |             jmp     #LMM_OP_MOVK                                          |
' |             long    $NNNN_NNNN                                            |
' |             mov     reg,tmp                                               |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_MOVK     rdlong  tmp,lmmPc
                add     lmmPc,#4
                jmp     #Lmm_Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "TJZ reg,#label" is coded as ...                               |
' |                                                                           |
' |             jmp     #LMM_OP_TJZ                                           |
' |             long    reg << 23 | @label                                    |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_TJZ      movi    Lmm_Opcode,#PASM_TJZ
                jmp     #LMM_OP_JMP_Fix

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "TJNZ reg,#label" is coded as ...                              |
' |                                                                           |
' |             jmp     #LMM_OP_TJNZ                                          |
' |             long    reg << 23 | @label                                    |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_TJNZ     movi    Lmm_Opcode,#PASM_TJNZ
                jmp     #LMM_OP_JMP_Fix

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "DJNZ reg,#label" is coded as ...                              |
' |                                                                           |
' |             jmp     #LMM_OP_DJNZ                                          |
' |             long    reg << 23 | @label                                    |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_DJNZ     movi    Lmm_Opcode,#PASM_DJNZ

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     Fixup LMM TJZ, TJNZ and DJNZ opcodes                                  |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_JMP_Fix  rdlong  tmp,lmmPc
                shr     tmp,#23
                movd    Lmm_Opcode,tmp
                movs    Lmm_Opcode,#LMM_OP_JMP
                jmp     #Lmm_Opcode

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     An LMM "PUSH_xxx" is coded as ...                                     |
' |                                                                           |
' |             jmp     #LMM_OP_PUSH_xxx                                      |
' |             long    $xxxx_xxxx                                            |
' |                                                                           |
' `---------------------------------------------------------------------------'

LMM_OP_PUSH_MEM mov     rhs,k_LccCode
                jmp     #Lmm_AddPush

LMM_OP_PUSH_ARG mov     rhs,argPtr
                jmp     #Lmm_AddPush

LMM_OP_PUSH_LOC mov     rhs,locPtr
                jmp     #Lmm_AddPush

LMM_OP_PUSH_NUM mov      rhs,#0

Lmm_AddPush     rdlong  lhs,lmmPc
                add     lhs,rhs
                sub     sp,#4
                wrlong  lhs,sp
                add     lmmPc,#4
                jmp     #Lmm_Fetch

' <<END IF>>

DAT

' *****************************************************************************
' *                                                                           *
' *     Constants                                                             *
' *                                                                           *
' *****************************************************************************

' <<DEFINE CONSTANTS>>

' <<IF KERNEL.LIBS>>
k_LccVm         long    @LccVm
k_LibPtrTable   long    @LibPtrTable
' <<END IF>>
k_LccCode       long    @LccCode

k_0000_0200     long    $0000_0200
k_FFFF_FFFF     long    $FFFF_FFFF

Minus23         long    -23
Mask23          long    $007F_FFFF
NaN             long    $7FFF_FFFF
Bit29           long    $2000_0000

' <<FINISH CONSTANTS>>

DAT

' *****************************************************************************
' *                                                                           *
' *     Virtual Machine Register                                              *
' *                                                                           *
' *****************************************************************************

' .---------------------------------------------------------------------------.
' |     Core registers which the LCC Virtual Machine depends upon             |
' `---------------------------------------------------------------------------'

argNxt          long    0
' <<IF NORAM>>
romPtr          long    0-0
' <<END IF>>

' .---------------------------------------------------------------------------.
' |     Registers which the running LMM will require access to                |
' `---------------------------------------------------------------------------'

lhs             long    0-0
rhs             long    0-0
ptr             long    0-0

' .---------------------------------------------------------------------------.
' |     Initialisation routine pre-loaded into overlay area                   |
' `---------------------------------------------------------------------------'

INITIALISE

GetArg          mov     $1F7,GetArg_Ret
                sub     $1F7,#1
                movd    :PtrGet,$1F7
                mov     $1F7,PAR
:PtrInc         add     $1F7,#0
:PtrGet         rdlong  0-0,$1F7
                add     :PtrInc,#4
GetArg_Ret      ret

INITIALISE_END

' .---------------------------------------------------------------------------.
' |     Ensure everything fits into the Cog                                   |
' `---------------------------------------------------------------------------'

' <<IF DEBUG>>
                fit     $1EF
' <<ELSE>>
                fit     $1F0
' <<END IF>>

' .---------------------------------------------------------------------------.
' |     Temporary registers                                                   |
' `---------------------------------------------------------------------------'

                org     INITIALISE

OVERLAY

opc             res     1

acc             res     1

lhsTop          res     1
rhsTop          res     1
accTop          res     1

signBit         res     1

' for floating point

expA            res     1
flagA           res     1
expB            res     1
flagB           res     1

lmmStack        res     2                       ' Must be last

OVERLAY_END

' .---------------------------------------------------------------------------.
' |     Esure everything still fits into the Cog                              |
' `---------------------------------------------------------------------------'

' <<IF DEBUG>>
                fit     $1EF
' <<ELSE>>
                fit     $1F0
' <<END IF>>

' .---------------------------------------------------------------------------.
' |     Temporary registers overlaid onto permanent ones                      |
' `---------------------------------------------------------------------------'

                org     lhs
                        lhsBot

                org     rhs
                        rhsBot

                org     acc
                        accBot

' .---------------------------------------------------------------------------.
' |     FP Temporary registers overlaid onto permanent / Temporary ones        |
' `---------------------------------------------------------------------------'

                org     lhs
                        fnumA

                org     rhs
                        fnumB

                org     lhsTop
                        manA

                org     rhsTop
                        manB

' .---------------------------------------------------------------------------.
' |     FP Routines which vector to others                                    |
' `---------------------------------------------------------------------------'

                org     PopLhs
                        PopFnumA

                org     PopLhs_Ret
                        PopFnumA_Ret

                org     PopLhsAndRhs
                        PopFnumAandFnumB

                org     PopLhsAndRhs_Ret
                        PopFnumAandFnumB_Ret

' <<IF DEBUG>>

' .---------------------------------------------------------------------------.
' |     Register address support for debugger when using overlay registers    |
' `---------------------------------------------------------------------------'

OVERLAY_opc     word    opc

' .---------------------------------------------------------------------------.
' |     SPR address support for debugger                                      |
' `---------------------------------------------------------------------------'

REG_PAR         word    PAR     ' $1F0
REG_CNT         word    CNT     ' $1F1
REG_INA         word    INA     ' $1F2
REG_INB         word    INB     ' $1F3
REG_OUTA        word    OUTA    ' $1F4
REG_OUTB        word    OUTB    ' $1F5
REG_DIRA        word    DIRA    ' $1F6
REG_DIRB        word    DIRB    ' $1F7
REG_CTRA        word    CTRA    ' $1F8
REG_CTRB        word    CTRB    ' $1F9
REG_FRQA        word    FRQA    ' $1FA
REG_FRQB        word    FRQB    ' $1FB
REG_PHSA        word    PHSA    ' $1FC
REG_PHSB        word    PHSB    ' $1FD
REG_VCFG        word    VCFG    ' $1FE
REG_VSCL        word    VSCL    ' $1FF

' <<END IF>>

CON

' *****************************************************************************
' *                                                                           *
' *     End Of LCC Virtual Machine                                            *
' *                                                                           *
' *****************************************************************************

' <<IF KERNEL.LIBS>>

CON

' *****************************************************************************
' *                                                                           *
' *     LMM Executed Library Code                                             *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Integer Maths Library                                                 *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB MPY 2>>

LIB_MPY_I32     jmp     #LMM_OP_CALL
                long    @LIB_PopLhsRhsClrSgn
                add     lmmPc,#2*4              ' ---.
                                                '    |
' <<LIB SMPY 2>>                                '    |
                                                '    |
LIB_MPY_S32     jmp     #LMM_OP_CALL            '    | +0
                long    @LIB_PopLhsRhsSetSgn    '    | +1
                jmp     #LMM_OP_CALL            ' <--' +2
                long    @LIBSUB_Mpy
                mov     lhs,accBot
                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB MPYBIG 2>>

LIB_MBG_I32     jmp     #LMM_OP_CALL
                long    @LIB_PopLhsRhsClrSgn
                add     lmmPc,#2*4              ' ---.
                                                '    |
' <<LIB SMPYBIG 2>>                             '    |
                                                '    |
LIB_MBG_S32     jmp     #LMM_OP_CALL            '    | +0
                long    @LIB_PopLhsRhsSetSgn    '    | +1
                jmp     #LMM_OP_CALL            ' <--' +2
                long    @LIBSUB_Mpy
                mov     lhs,accTop
                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB MOD 2>>

LIB_MOD_I32     jmp     #LMM_OP_CALL
                long    @LIB_PopLhsRhsClrSgn
                add     lmmPc,#2*4              ' ---.
                                                '    |
' <<LIB SMOD 2>>                                '    |
                                                '    |
LIB_MOD_S32     jmp     #LMM_OP_CALL            '    | +0
                long    @LIB_PopLhsRhsSetSgn    '    | +1
                jmp     #LMM_OP_CALL            ' <--' +2
                long    @LIBSUB_Div
                jmp     #LMM_OP_JMP
                long    @LIB_FixSignLhsPush

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB DIV 2>>

LIB_DIV_I32     jmp     #LMM_OP_CALL
                long    @LIB_PopLhsRhsClrSgn
                add     lmmPc,#2*4              ' ---.
                                                '    |
' <<LIB SDIV 2>>                                '    |
                                                '    |
LIB_DIV_S32     jmp     #LMM_OP_CALL            '    | +0
                long    @LIB_PopLhsRhsSetSgn    '    | +1
                jmp     #LMM_OP_CALL            ' <--' +2
                long    @LIBSUB_Div
                mov     lhs,acc

LIB_FixSignLhsPush

                shr     signBit,#1      WC, NR
    IF_C        neg     lhs,lhs
                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

LIB_PopLhsRhsClrSgn

                call    #PopLhsAndRhs
                mov     signBit,#0
                jmp     #LMM_OP_RET

LIB_PopLhsRhsSetSgn

                call    #PopLhsAndRhs
                abs     lhs,lhs         WC      ' Sets C if was -Ve
                muxc    signBit,#1
                abs     rhs,rhs         WC      ' Sets C if was -Ve
        IF_C    xor     signBit,#1
                jmp     #LMM_OP_RET

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB MIN 2>>

LIB_MIN_I32

' <<LIB HIGHEST 2>>

LIB_HGH_I32     call    #PopLhsAndRhs
                cmp     rhs,lhs         WZ, WC
        IF_A    mov     lhs,rhs
                jmp     #Push

' <<LIB MAX 2>>

LIB_MAX_I32

' <<LIB LOWEST 2>>

LIB_LOW_I32     call    #PopLhsAndRhs
                cmp     rhs,lhs         WZ, WC
        IF_B    mov     lhs,rhs
                jmp     #Push

' <<LIB SMIN 2>>

LIB_MIN_S32

' <<LIB SHIGHEST 2>>

LIB_HGH_S32     call    #PopLhsAndRhs
                cmps    rhs,lhs         WZ, WC
        IF_A    mov     lhs,rhs
                jmp     #Push

' <<LIB SMAX 2>>

LIB_MAX_S32

' <<LIB SLOWEST 2>>

LIB_LOW_S32     call    #PopLhsAndRhs
                cmps    rhs,lhs         WZ, WC
        IF_B    mov     lhs,rhs
                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB ABS 1>>

LIB_ABS_S32   call      #PopLhs
              abs       lhs,lhs
              jmp       #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

'               http://forums.parallax.com/forums/default.aspx?f=25&p=1&m=269640

' <<LIB SQR 1>>

LIB_SQR_S32     call    #PopPtr
                abs     ptr,ptr

                mov     lhs,#0                  ' reset lhs
:LIB_Sqr_Loop   or      lhs,k_4000_0000         ' set trial bit
                cmpsub  ptr,lhs  wc             ' subtract lhs from input if fits
                sumnc   lhs,k_4000_0000         ' cancel trial bit, set lhs bit if fit
                shr     lhs,#1                  ' shift lhs down
  IF_ALWAYS     ror     k_4000_0000,#2  WC      ' shift k_4000_0000 down (wraps on last iteration)
  IF_NC         sub     lmmPc,#6*4              ' loop until k_4000_0000 restored

                jmp     #Push

' .---------------------------------------------------------------------------.
' |     32 x 32 to 64 Multiply                                                |
' `---------------------------------------------------------------------------'

LIBSUB_Mpy      mov     lhsTop,#0
                mov     accBot,#0
                mov     accTop,#0

:LMM_Mpy_Loop   shr     rhs,#1          WC, WZ  ' <--. -7
                                                '    :
        IF_C    add     accTop,lhsTop           '    | -6
        IF_C    add     accBot,lhsBot   WC      '    | -5
        IF_C    add     accTop,#1               '    | -4
                                                '    :
                shl     lhsBot,#1       WC      '    | -3
                rcl     lhsTop,#1               '    | -2
                                                '    :
        IF_NZ   sub     lmmPc,#7*4              ' ---' -1

                shr     signBit,#1      WC, NR
        IF_C    neg     accBot,accBot   WZ
        IF_C    neg     accTop,accTop
  IF_C_AND_NZ   sub     accTop,#1

                jmp     #LMM_OP_RET

' .---------------------------------------------------------------------------.
' |     32 / 32 = 32 Divide and Modulus                                       |
' `---------------------------------------------------------------------------'

LIBSUB_Div      mov     acc,#0
                mov     accTop,#1

:LIB_Div_Loop   and     rhs,k_4000_0000 WZ, NR  ' <--  -4
        IF_Z    shl     rhs,#1                  '    | -3
        IF_Z    add     accTop,#1               '    | -2
        IF_Z    sub     lmmPc,#4*4              ' ---' -1

:LIB_Div_Sub    shl     acc,#1                  ' <--. -7
                cmps    lhs,rhs         WZ, WC, NR   ' -6
        IF_AE   sub     lhs,rhs                 '    | -5
        IF_AE   or      acc,#1                  '    | -4
                shr     rhs,#1                  '    | -3
                sub     accTop,#1       WZ      '    | -2
        IF_NZ   sub     lmmPc,#7*4              ' ---' -1

'               acc   = result
'               lhs   = remainder

                jmp     #LMM_OP_RET

' *****************************************************************************
' *                                                                           *
' *     End of Integer Maths Library                                          *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *    System Function Library                                                *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB WAITCNT 1>>

LIB_WaitCnt     call    #PopLhs
                ' <<IF DEBUG>>
                add     lhs,CNT
                ' <<END IF>>
                waitcnt lhs,#0
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB WAITPEQ 3>>

LIB_WaitPeq     call    #PopLhsAndRhsAndPtr
                test    ptr,#1                  WC
                waitpeq lhs,rhs
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB WAITPNE 3>>

LIB_WaitPne     call    #PopLhsAndRhsAndPtr
                test    ptr,#1                  WC
                waitpne lhs,rhs
                jmp     #Fetch

' *****************************************************************************
' *                                                                           *
' *     End of System Library                                                 *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Cog / Task Handling Library                                           *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB COGID 0>>

LIB_CogId

' <<LIB TASKID 0>>

LIB_TaskId      cogid   lhs
                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB COGSTOP 1>>

LIB_CogStop

' <<LIB TASKSTOP 1>>

LIB_TaskStop    call    #PopLhs
                cogstop lhs
                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' --<<LIB COGNEW 2>>
'
'LIB_CogNew      call    #PopLhsAndRhs
'
'                jmp     #LMM_OP_MOVK
'                long    $0000_FFFC
'
'                and     rhs,tmp                         ' rhs = PAR
'                shl     rhs,#18
'
'                and     lhs,tmp                         ' lhs = ADR
'                shl     lhs,#4
'
'                or      lhs,rhs
'                or      lhs,#%1000
'                coginit lhs                     WC
'        IF_C    mov     lhs,k_FFFF_FFFF
'                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' --<<LIB COGINIT 3>>
'
'LIB_CogInit     call    #PopLhsAndRhsAndPtr
'
'                jmp     #LMM_OP_MOVK
'                long    $0000_FFFC
'
'                and     ptr,tmp                         ' ptr = PAR
'                shl     ptr,#18
'
'                and     rhs,tmp                         ' rhs = ADR
'                shl     rhs,#4
'
'                and     lhs,#7                          ' lhs = COG
'
'                or      lhs,ptr
'                or      lhs,rhs
'                coginit lhs                     WC
'        IF_C    mov     lhs,k_FFFF_FFFF
'                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     Start a C function running in another Cog. The commands for this are  |
' |                                                                           |
' |         TaskInit( cogNumber, &function, &stack[0], stackSize );           |
' |         TaskNew( &function, &stack[0], stackSize );                       |
' |                                                                           |
' |     Note that this is quite a bit more complicated than pulling stuff off |
' |     the stack and just using it. A launched LccVm Cog expects to be given |
' |     a parameter block pointed to by PAR which indicates the environment   |
' |     it will be working within and this has to be built correctly.         |
' |                                                                           |
' `---------------------------------------------------------------------------'

'       On entry ...
'
'               |--------|                              |----------|
'               | pc     | ---------------------------> |   LCC    |
'               | stkAdr | ---.                         | bytecode |
'       sp ---> | stkSiz |    |                         |----------|
'               `--------'    |
'                             |                         .----------.
'                             `-----------------------> | stack[n] |
'                                                       |          |
'                                                       |          |
'                                                       `----------'
'       After processing ...
'
'       .--------------.--------------.---.-----.
'       |      PAR     |      ADR     | 1 | 000 |  Parameter for CogInit
'       `-------.------^-------.------^---^-----'
'               |              |
'               |              |                        |----------|
'       .-------'              `----------------------> | LmmVm    |
'       |                                     .-------> |----------|
'       |       | XXXXXX |                    |
'       |       |--------|                    |         |----------|
'       `-----> | pc     | -------------------|-------> |   LCC    |
'               | argPtr | ---.               |         | bytecode |
'               | sp     | ---|---.           |         |----------|
'               | locPtr | ---|---.           |
'       sp ---> | lmmBas | ---|---|-----------'         .----------.
'               `--------'    |   |                     | stack[n] |
'                             |   |                     |          |
'                             |   `-------------------> |   0000   |
'                             `-----------------------> |   0000   |
'                                                       `----------'
'       On exit ...
'
'               | XXXXXX |
'               |--------|
'       sp ---> | result |
'               `--------'

' <<LIB TASKINIT 3>>

LIB_TaskInit    call    #PopLhsAndRhsAndPtr     ' lhs = cogNumber
                mov     accTop,lhs              ' rhs = pc
                and     accTop,#%0111           ' ptr = stack address

                ' We now need to jiggle things about because there were
                ' four arguments passed in and we need things set how
                ' TaskNew expects them.

                mov     lhs,rhs                 ' lhs = pc
                mov     rhs,ptr                 ' rhs = stack address
                mov     ptr,argPtr              ' ptr = stack size
                sub     ptr,#3*4
                rdlong  ptr,ptr
                add     lmmPc,#2*2              ' ---. -1
                                                '    :
' <<LIB TASKNEW 3>>                             '    :
                                                '    :
LIB_TaskNew     call    #PopLhsAndRhsAndPtr     '    | +0
                mov     accTop,#%1000           '    | +1
                                                '    :
                mov     tmp,#0                  ' <--' +2


                ' lhs = absolute hub address of LCC bytecode to execute
                ' rhs = absolute hub address of bottom of stack
                ' ptr = size of stack ( number of longs )

                shl     ptr,#2                  ' Number of bytes
                add     rhs,ptr                 ' Add to base
                andn    rhs,#3                  ' Ensure it's long aligned
                sub     rhs,#4                  ' Point to top of stack

                ' Point to whre parameter block starts on stack

                sub     sp,#5*4

                ' Determine the configuration long for CogInit
                '
                ' PPPP PPPP PPPP PP00                       PAR
                '                  AA AAAA AAAA AAAA 00     ADR
                ' pppp pppp pppp ppaa aaaa aaaa aaaa nccc

                mov     acc,sp                  ' PAR = param block
                shl     acc,#14
                add     acc,k_LccVm             ' ADR = @LccVm
                add     acc,hubOffset
                shl     acc,#2
                or      acc,accTop              ' NEW = ?, COG = ?

                ' Build the parameter block

                ' lhs = absolute hub address of LCC bytecode to execute
                ' rhs = absolute hub address of long at top stack

                wrlong  tmp,rhs                 ' Zero the Return Pc

                wrlong  lhs,sp                  ' paramPc
                add     sp,#4

                wrlong  rhs,sp                  ' paramArgPtr
                add     sp,#4

                sub     rhs,#4                  ' Zero the ENT count
                wrlong  tmp,rhs

                wrlong  rhs,sp                  ' paramSp
                add     sp,#4

                wrlong  rhs,sp                  ' paramLocPtr
                add     sp,#4

                mov     ptr,k_LccVm             ' paramLccBase
                add     ptr,hubOffset

                wrlong  ptr,sp

                ' Initiate the CogInit to get another LccVm running

                coginit acc             WR, WC  ' Start the cog, get CogId
        IF_C    mov     acc,k_FFFF_FFFF         ' If failed, -1

                ' Wait for the paramPc to be cleared

                 sub     sp,#4*4                ' Point to paramPc

:Loop            rdlong  tmp,sp          WZ     ' <--. -2
         IF_NZ   sub     lmmPc,#2*4             ' ---' -1

                 add     sp,#4*4                ' Point to where return goes

                ' Return the result

                wrlong  acc,sp                  ' Saves moving to lhs then push 
                jmp     #Fetch

' *****************************************************************************
' *                                                                           *
' *     End of Cog Handling Library                                           *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Thread Handling Library                                               *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |     void Threads( &threadArray, threadArraySize );                        |
' `---------------------------------------------------------------------------'

' <<LIB THREADS 1>>

LIB_Threads     call      #PopLhs

                jmp       #Fetch
              
' .---------------------------------------------------------------------------.
' |     void ThreadNew( &function, &stack[0], stackSize );                    |
' `---------------------------------------------------------------------------'

' <<LIB THREADNEW 3>>

LIB_ThreadNew   call      #PopLhsAndRhsAndPtr

                jmp       #Fetch
                
' .---------------------------------------------------------------------------.
' |     void ThreadStop();                                                    |
' `---------------------------------------------------------------------------'

' <<LIB THREADSTOP 0>>

LIB_ThreadStop

                jmp       #Fetch

' *****************************************************************************
' *                                                                           *
' *     End of Thread Handling Library                                        *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Lock Handling Library                                                 *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB LOCKNEW 0>>

LIB_LockNew   locknew   lhs                     WC
        IF_C  mov       lhs,k_FFFF_FFFF
              jmp       #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB LOCKSET 1>>

LIB_LockSet   call      #PopLhs
              lockset   lhs                     WC
              muxc      lhs,k_FFFF_FFFF
              jmp       #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB LOCKCLR 1>>

LIB_LockClr   call      #PopLhs
              lockclr   lhs                     WC
              muxc      lhs,k_FFFF_FFFF
              jmp       #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB LOCKRET 1>>

LIB_LockRet   call      #PopLhs
              lockret   lhs
              jmp       #Fetch

' *****************************************************************************
' *                                                                           *
' *     End of Lock Handling Library                                          *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Serial Comms Library                                                  *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |           TxText( pin, baud, strPtr ) -Ve baud = invert signal            |
' `---------------------------------------------------------------------------'

' <<LIB TXTEXT 3>>

LIB_TxText      call    #PopLhs
                mov     opc,lhs

                jmp     #LMM_OP_CALL
                long    @LIB_SetBaudAndPin

                ' rhs = pin mask, acc = bit time, ptr = data, signBit = invert

:Loop           rdbyte  ptr,opc                 WZ      ' <--. -6
        IF_Z    jmp     #Fetch                          '    | -5
                                                        '    :
                jmp     #LMM_OP_CALL                    '    | -4
                long    @LIBSUB_TxByte                  '    | -3
                                                        '    :
                add     opc,#1                          '    | -2
                sub     lmmPc,#6*4                      ' ---' -1

' .---------------------------------------------------------------------------.
' |           TxByte( pin, baud, data ) -Ve baud = invert signal              |
' `---------------------------------------------------------------------------'

' <<LIB TXBYTE 3>>

LIB_TxByte

' <<LIB TXCHAR 3>>

LIB_TxChar      call    #PopPtr

                jmp     #LMM_OP_CALL
                long    @LIB_SetBaudAndPin

                ' rhs = pin mask, acc = bit time, ptr = data, signBit = invert

                jmp     #LMM_OP_CALL
                long    @LIBSUB_TxByte

                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |           TxByte( pin, baud, data ) -Ve baud = invert signal              |
' `---------------------------------------------------------------------------'

' <<LIB TXRAW 3>>

LIB_TxRaw       call    #PopPtr

                jmp     #LMM_OP_CALL
                long    @LIB_SetBaudAndPin

                ' rhs = pin mask, acc = bit time, ptr = data, signBit = invert

                jmp     #LMM_OP_CALL
                long    @LIBSUB_TxRaw

                jmp     #Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' |     Send a byte. Note that we have to do some messing about to cope with  |
' |     \n for newline. LCC converts this to $0A (LF) so if we see an LF we   |
' |     change that to a $0D (CR) and whenever we send a $0D (CR) we always   |
' |     send a $0A (LF) after it. This probably causes some problems with \r  |
' |     as that will also go out as a CR-LF pair but it's reasonable to say   |
' |     that sending anything other than CR-LF pairs is the unusual case.     |
' |                                                                           |
' |     LIBSUB_TxRaw can be used to send LF without conversion but CR will    |
' |     be changed to a CR-LF pair. This could be altered but doesn't seem    |
' |     to be worth the effort.                                               |
' |                                                                           |
' |     In definition here, 'normal polarity' means that a 1 bit will be      |
' |     sent out as a +Ve signal. This is suitable for an output pin which    |
' |     goes direct to a PC. If to a MAX232 then the signal will need to be   |
' |     inverted. This is done automatically for the PropPlug TX,             |
' |                                                                           |
' `---------------------------------------------------------------------------'

LIBSUB_TxByte   cmp     ptr,#$0A        WZ
        IF_Z    mov     ptr,#$0D

LIBSUB_TxRaw    test signBit,#1         WC

                muxc    OUTA,rhs
                or      DIRA,rhs

:LIB_TxLf       mov     lhsTop,ptr              ' Remember what we sent <--. -16
                                                '                          :
                shl     ptr,#1                  ' Set the start bit        | -15
                xor     ptr,#$1FF               ' Assume normal polarity   | -14
        IF_C    xor     ptr,k_FFFF_FFFF         ' Invert as necessay       | -13
                                                '                          :
                mov     tmp,#10                 '                          | -12
                                                '                          :
                mov     accTop,acc              '                          | -11
                add     accTop,CNT              '                          | -10
                                                '                          :
:LIB_TxLoop     shr     ptr,#1          WC      ' <--. -5                  | -9
                muxc    OUTA,rhs                '    | -4                  | -8
                                                '    :                     :
                waitcnt accTop,acc              '    | -3                  | -7
                                                '    :                     :
                sub     tmp,#1          WZ      '    | -2                  | -6
        IF_NZ   sub     lmmPc,#5*4              ' ---' -1                  | -5
                                                '                          :
                cmp     lhsTop,#$0D     WZ      ' Did we send CR ?         | -4
        IF_NZ   jmp     #LMM_OP_RET             ' No                       | -3
                                                '                          :
                mov     ptr,#$0A                ' Yes, send LF             | -2
                sub     lmmPc,#16*4             '                       ---' -1

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB RXBYTE 2>>

LIB_RxByte

' <<LIB RXCHAR 2>>

LIB_RxChar      jmp     #LMM_OP_CALL
                long    @LIB_SetBaudAndPin

                ' rhs = pin mask, acc = bit time, signBit = invert

                andn    DIRA,rhs                ' Make pin an input

                mov     accTop,acc              ' Time to halfway into start
                shr     accTop,#1

                mov     tmp,#9                  ' Read start + 8 data bits

                test    signBit,#1      WZ      ' Wait for start bit

        IF_Z    waitpne rhs,rhs
                waitpeq rhs,rhs
        IF_NZ   waitpne rhs,rhs

                add     accTop,CNT              ' Wait until mid bit

:LIB_RxLoop     waitcnt accTop,acc              '                    <--. -6
                                                '                       :
                test    rhs,INA         WC      ' Sample bit            | -5
                shr     lhs,#1                  '                       | -4
                muxc    lhs,#$80                '                       | -3
                                                '                       :
                sub     tmp,#1          WZ      ' Got all bits ?        | -2
        IF_NZ   sub     lmmPc,#6*4              ' No                 ---' -1

                test    signBit,#1      WC      ' Invert as required
        IF_NC   xor     lhs,#$FF

                jmp     #Push

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

LIB_SetBaudAndPin

                call    #PopLhs                 ' Baud rate
                abs     rhs,lhs         WC
                muxc    signBit,#1              ' Flag if inverted (lsb)

                rdlong  lhs,#0                  ' CLKFREQ
                jmp     #LMM_OP_CALL
                long    @LIBSUB_Div             ' acc := CLKFREQ / Baud

                call    #PopLhs                 ' Get pin
                mov     rhs,#1
                shl     rhs,lhs

                cmp     lhs,#28        WC, WZ   ' Invert if via PropPlug etc
        IF_AE   xor     signBit,#1

                jmp     #LMM_OP_RET             ' Finished

' *****************************************************************************
' *                                                                           *
' *     End of Serial Comms Library                                           *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Memory Manipulation Library                                           *
' *                                                                           *
' *****************************************************************************

DAT

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

' <<LIB BYTEFILL 3>>

LIB_ByteFill

' <<LIB CHARFILL 3>>

LIB_CharFill            call    #PopLhsAndRhsAndPtr     ' lhs = adr
                        wrbyte  ptr,lhs                 ' rhs = count  <--. -4
                        add     lhs,#1                  ' ptr = value     | -3
                        sub     rhs,#1          WZ      '                 | -2
              IF_NZ     sub     lmmPc,#4*4              ' ----------------' -1
                        jmp     #Fetch

' <<LIB WORDFILL 3>>

LIB_WordFill            call    #PopLhsAndRhsAndPtr     ' lhs = adr
                        wrword  ptr,lhs                 ' rhs = count  <--. -4
                        add     lhs,#2                  ' ptr = value     | -3
                        sub     rhs,#1          WZ      '                 | -2
              IF_NZ     sub     lmmPc,#4*4              ' ----------------' -1
                        jmp     #Fetch

' <<LIB LONGFILL 3>>

LIB_LongFill

' <<LIB REALFILL 3>>

LIB_RealFill            call    #PopLhsAndRhsAndPtr     ' lhs = adr
                        wrlong  ptr,lhs                 ' rhs = count  <--. -4
                        add     lhs,#4                  ' ptr = value     | -3
                        sub     rhs,#1          WZ      '                 | -2
              IF_NZ     sub     lmmPc,#4*4              ' ----------------' -1
                        jmp     #Fetch

' .---------------------------------------------------------------------------.
' |                                                                           |
' `---------------------------------------------------------------------------'

LIB_MOV_I08

' <<LIB BYTEMOVE 3>>

LIB_ByteMove

' <<LIB CHARMOVE 3>>

LIB_CharMove            mov     tmp,#0                  ' Bytes
                        jmp     #LMM_OP_JMP
                        long    @LIB_XxxxMove

' <<LIB WORDMOVE 3>>

LIB_WordMove            mov     tmp,#1                  ' Words
                        jmp     #LMM_OP_JMP
                        long    @LIB_XxxxMove

' <<LIB LONGMOVE 3>>

LIB_LongMove

' <<LIB REALMOVE 3>>

LIB_RealMove            mov     tmp,#2                  ' Longs

LIB_XxxxMove            Call    #PopLhsAndRhsAndPtr     ' lhs = dstAdr
                        shl     ptr,tmp                 ' rhs = srcAdr
                                                        ' ptr = byte count

:Loop                   rdbyte  tmp,rhs                 ' <--. -6
                        add     rhs,#1                  '    | -5
                        sub     ptr,#1          WZ      '    | -4
                        wrbyte  tmp,lhs                 '    | -3
                        add     lhs,#1                  '    | -2
              IF_NZ     sub     lmmPc,#6*4              ' ---' -1

                        jmp     #Fetch

' *****************************************************************************
' *                                                                           *
' *     End of Memory Manipulation Library                                    *
' *                                                                           *
' *****************************************************************************

CON

' *****************************************************************************
' *                                                                           *
' *     Floating Point Library                                                *
' *                                                                           *
' *****************************************************************************
' *                                                                           *
' *     Based upon the MIT licensed "Float32Full" code from the Propeller     *
' *     Tool 1.2 Library Directory by Carl Thompson, Micromega Corporation.   *
' *                                                                           *
' *     Original credits and comments from "Full32Full.Spin" ...              *
' *                                                                           *
' *        IEEE 754 compliant 32-bit floating point math routines.            *
' *                                                                           *
' *        Author   Cam Thompson, Micromega Corporation                       *
' *                                                                           *
' *        Copyright (c) 2006-2007 Parallax, Inc.                             *
' *        See end of file for terms of use.                                  *
' *'                                                                          *
' *        @version V1.4 - September 25, 2007                                 *
' *                                                                           *
' *****************************************************************************

  SignFlag      = $1
  ZeroFlag      = $2
  NaNFlag       = $8

DAT

' .---------------------------------------------------------------------------.
' |     Convert Signed 32 to Float                                            |
' `---------------------------------------------------------------------------'

'------------------------------------------------------------------------------
' _FFloat  fnumA = float(fnumA)
' changes: fnumA, flagA, expA, manA
'------------------------------------------------------------------------------

' <<LIB FLOAT 1>>

LIB_S2F_F32

_FFloat                 call    #PopFnumA
                        mov     flagA, fnumA            ' get integer value
                        mov     fnumA, #0               ' set initial result to zero
                        abs     manA, flagA wz          ' get absolute value of integer
          if_z          jmp     #Push            ' if zero, exit
                        shr     flagA, #31              ' set sign flag
                        mov     expA, #31               ' set initial value for exponent
:normalize              shl     manA, #1 wc             ' normalize the mantissa
          if_nc         sub     expA, #1                ' adjust exponent
          if_nc         sub     lmmPc,#3*4
                        rcr     manA, #1                ' justify mantissa
                        shr     manA, #2
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Pack                  ' pack and exit
                        jmp     #Push

' .---------------------------------------------------------------------------.
' |     Convert Float to Signed 32                                            |
' `---------------------------------------------------------------------------'


'------------------------------------------------------------------------------
' _FTrunc  fnumA = fix(fnumA)
' _FRound  fnumA = fix(round(fnumA))
' changes: fnumA, flagA, expA, manA, t1
'------------------------------------------------------------------------------

' <<LIB ROUND 1>>

LIB_ROUND               mov     tmp, #1                  ' set for rounding
                        add     lmmPc,#1*4

' <<LIB TRUNC 1>>

LIB_TRUNC

LIB_F2S_S32             mov     tmp, #0                  ' set for no rounding

                        call    #PopFnumA

fix                     jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack          ' unpack floating point value
          if_c          jmp     #Push                   ' check for NaN
                        shl     manA, #2                ' left justify mantissa
                        mov     fnumA, #0               ' initialize result to zero
                        neg     expA, expA              ' adjust for exponent value
                        add     expA, #30 wz
                        cmps    expA, #32 wc
          if_nc_or_z    jmp     #Push
                        shr     manA, expA

                        add     manA, tmp                ' round up 1/2 lsb
                        shr     manA, #1

                        test    flagA, #signFlag wz     ' check sign and exit
                        sumnz   fnumA, manA
_FTrunc_ret
_FRound_ret             jmp     #Push

' .---------------------------------------------------------------------------.
' |     Add and Subtract                                                      |
' `---------------------------------------------------------------------------'

'------------------------------------------------------------------------------
' _FAdd    fnumA = fnumA + fNumB
' _FAddI   fnumA = fnumA + {Float immediate}
' _FSub    fnumA = fnumA - fNumB
' _FSubI   fnumA = fnumA - {Float immediate}
' changes: fnumA, flagA, expA, manA, fnumB, flagB, expB, manB, t1
'------------------------------------------------------------------------------

' <<LIB FSUB 2>>

LIB_SUB_F32

_FSub                   call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack
                         xor     fnumA, k_8000_0000            ' negate B
                        jmp     #LMM_OP_JMP
                        long    @_FXXXX

' <<LIB FADD 2>>

LIB_ADD_F32

_FAdd                   call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack

_FXXXX                  jmp     #LMM_OP_CALL
                        long    @MovFnumBfromFnumA

                        call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack
          if_c_or_z     jmp     #Push              ' check for NaN or B = 0

                        test    flagA, #SignFlag wz     ' negate A mantissa if negative
          if_nz         neg     manA, manA
                        test    flagB, #SignFlag wz     ' negate B mantissa if negative
          if_nz         neg     manB, manB

                        mov     tmp, expA                ' align mantissas
                        sub     tmp, expB
                        abs     tmp, tmp
                        max     tmp, #31
                        cmps    expA, expB wz,wc
          if_nz_and_nc  sar     manB, tmp
          if_nz_and_c   sar     manA, tmp
          if_nz_and_c   mov     expA, expB

                        add     manA, manB              ' add the two mantissas
                        cmps    manA, #0 wc, nr         ' set sign of result
          if_c          or      flagA, #SignFlag
          if_nc         andn    flagA, #SignFlag
                        abs     manA, manA              ' pack result and exit
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Pack
_FSubI_ret
_FSub_ret
_FAddI_ret
_FAdd_ret               jmp     #Push

MovFnumBfromFnumA       mov     FnumB,FnumA
                        mov     manB,manA
                        mov     expB,expA
                        mov     flagB,flagA
                        jmp     #LMM_OP_RET


' .---------------------------------------------------------------------------.
' |     Multiply                                                              |
' `---------------------------------------------------------------------------'

'------------------------------------------------------------------------------
' _FMul    fnumA = fnumA * fNumB
' _FMulI   fnumA = fnumA * {Float immediate}
' changes: fnumA, flagA, expA, manA, fnumB, flagB, expB, manB, t1, t2
'------------------------------------------------------------------------------

' <<LIB FMPY 2>>

LIB_MPY_F32

_FMul                   call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack
                        jmp     #LMM_OP_CALL
                        long    @MovFnumBfromFnumA
                        call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack



                        xor     flagA, flagB            ' get sign of result
                        add     expA, expB              ' add exponents
                        mov     tmp, #0                  ' t2 = upper 32 bits of manB
                        mov     opc, #32                 ' loop counter for multiply
                        shr     manB, #1 wc             ' get initial multiplier bit

:multiply if_c          add     tmp, manA wc             ' 32x32 bit multiply
                        rcr     tmp, #1 wc
                        rcr     manB, #1 wc
                        sub     opc,#1 wz
             if_NZ      sub     lmmPc,#5*4

                        shl     tmp, #3                  ' justify result and exit
                        mov     manA, tmp
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Pack
_FMulI_ret
_FMul_ret               jmp     #Push


' .---------------------------------------------------------------------------.
' |     Divide                                                                |
' `---------------------------------------------------------------------------'

'------------------------------------------------------------------------------
' _FDiv    fnumA = fnumA / fNumB
' _FDivI   fnumA = fnumA / {Float immediate}
' changes: fnumA, flagA, expA, manA, fnumB, flagB, expB, manB, t1, t2
'------------------------------------------------------------------------------

' <<LIB FDIV 2>>

LIB_DIV_F32

_FDiv                   call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack
                        jmp     #LMM_OP_CALL
                        long    @MovFnumBfromFnumA
                        call    #PopFnumA
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Unpack

                        xor     flagA, flagB            ' get sign of result
                        sub     expA, expB              ' subtract exponents
                        mov     tmp, #0                  ' clear quotient
                        mov     opc, #30                 ' loop counter for divide

:divide                 shl     tmp, #1                  ' divide the mantissas
                        cmps    manA, manB wz,wc
          if_z_or_nc    sub     manA, manB
          if_z_or_nc    add     tmp, #1
                        shl     manA, #1
                        sub     opc,#1 WZ
              IF_NZ     sub     lmmPc,#7*4

                        mov     manA, tmp                ' get result and exit
                        jmp     #LMM_OP_CALL
                        long    @LIBSUB_Pack
_FDivI_ret
_FDiv_ret               jmp     #Push


' .---------------------------------------------------------------------------.
' |     Number Packing                                                        |
' `---------------------------------------------------------------------------'

'------------------------------------------------------------------------------
' input:   flagA        fnumA flag bits (Nan, Infinity, Zero, Sign)
'          expA         fnumA exponent (no bias)
'          manA         fnumA mantissa (aligned to bit 29)
' output:  fnumA        32-bit floating point value
' changes: fnumA, flagA, expA, manA
'------------------------------------------------------------------------------

LIBSUB_Pack             cmp     manA, #0 wz             ' check for zero
          if_z          mov     expA, #0
          if_z          jmp     #LMM_OP_JMP
                        long    @pkexit1

:normalize              shl     manA, #1 wc             ' normalize the mantissa
          if_nc         sub     expA, #1                ' adjust exponent
          if_nc         sub     lmmPc,#3*4

                        add     expA, #2                ' adjust exponent
                        add     manA, #$100 wc          ' round up by 1/2 lsb
          if_c          add     expA, #1

                        add     expA, #127              ' add bias to exponent
                        mins    expA, Minus23
                        maxs    expA, #255

                        cmps    expA, #1 wc             ' check for subnormals
          if_nc         jmp     #LMM_OP_JMP
                        long    @pkexit1

:subnormal              or      manA, #1                ' adjust mantissa
                        ror     manA, #1

                        neg     expA, expA
                        shr     manA, expA
                        mov     expA, #0                ' biased exponent = 0

pkexit1                 mov     fnumA, manA             ' bits 22:0 mantissa
                        shr     fnumA, #9
                        movi    fnumA, expA             ' bits 23:30 exponent
                        shl     flagA, #31
                        or      fnumA, flagA            ' bit 31 sign
                        jmp     #LMM_OP_RET

'------------------------------------------------------------------------------
' input:   fnumA        32-bit floating point value
' output:  flagA        fnumA flag bits (Nan, Infinity, Zero, Sign)
'          expA         fnumA exponent (no bias)
'          manA         fnumA mantissa (aligned to bit 29)
'          C flag       set if fnumA is NaN
'          Z flag       set if fnumA is zero
' changes: fnumA, flagA, expA, manA
'------------------------------------------------------------------------------

LIBSUB_Unpack           mov     flagA, fnumA            ' get sign
                        shr     flagA, #31
                        mov     manA, fnumA             ' get mantissa
                        and     manA, Mask23
                        mov     expA, fnumA             ' get exponent
                        shl     expA, #1
                        shr     expA, #24 wz
          if_z          jmp     #LMM_OP_JMP
                        long    @zeroSubnormal         ' check for zero or subnormal
                        cmp     expA, #255 wz           ' check if finite
          if_nz         jmp     #LMM_OP_JMP
                        long    @finite
                        mov     fnumA, NaN              ' no, then return NaN
                        mov     flagA, #NaNFlag
                        jmp     #LMM_OP_JMP
                        long    @unpexit2

zeroSubnormal           or      manA, expA wz,nr        ' check for zero
          if_nz         jmp     #LMM_OP_JMP
                        long    @subnorm
                        or      flagA, #ZeroFlag        ' yes, then set zero flag
                        neg     expA, #150              ' set exponent and exit
                        jmp     #LMM_OP_JMP
                        long    @unpexit2

subnorm                 shl     manA, #7                ' fix justification for subnormals
subnorm2                test    manA, Bit29 wz
          if_nz         jmp     #LMM_OP_JMP
                        long    @unpexit1
                        shl     manA, #1
                        sub     expA, #1
                        jmp     #LMM_OP_JMP
                        long    @subnorm2

finite                  shl     manA, #6                ' justify mantissa to bit 29
                        or      manA, Bit29             ' add leading one bit

unpexit1                sub     expA, #127              ' remove bias from exponent
unpexit2                test    flagA, #NaNFlag wc      ' set C flag
                        cmp     manA, #0 wz             ' set Z flag
_Unpack_ret             jmp     #LMM_OP_RET

' *****************************************************************************
' *                                                                           *
' *     End of Floating Point Library                                         *
' *                                                                           *
' *****************************************************************************

' <<END IF>>

' <<IF DEBUG>>

CON

' *****************************************************************************
' *                                                                           *
' *     Printable names of opcodes                                            *
' *                                                                           *
' *****************************************************************************

DAT

' <<INSERT OPCNAMES>>

OpcNames        byte    $C0, "JPI", 0
                byte    $C1, "ENT.LOZ", 0
                byte    $C2, "INC.I32", 0
                byte    $00, "?",0

' <<FINISH OPCNAMES>>

CON

' *****************************************************************************
' *                                                                           *
' *     End of Printable names of opcodes                                     *
' *                                                                           *
' *****************************************************************************

' <<IF KERNEL.LIBS>>

CON

' *****************************************************************************
' *                                                                           *
' *     Printable names of libraries                                          *
' *                                                                           *
' *****************************************************************************

DAT

' <<INSERT LIBNAMES>>

LibNames        byte    0

' <<FINISH LIBNAMES>>

CON

' *****************************************************************************
' *                                                                           *
' *     End of Printable names of libraries                                   *
' *                                                                           *
' *****************************************************************************

' <<END IF>>

' <<END IF>>

' <<IF DEBUG>>
' <<OR TEST>>
' <<IF KERNEL.VARS>>

CON

' *****************************************************************************
' *                                                                           *
' *     Printable names of variables                                          *
' *                                                                           *
' *****************************************************************************

DAT

' <<INSERT VARNAMES>>

VarNames        byte    4, $00, $04, "count", 0
                byte    0

' <<FINISH VARNAMES>>

CON

' *****************************************************************************
' *                                                                           *
' *     End of Printable names of variables                                   *
' *                                                                           *
' *****************************************************************************

' <<END IF>>
' <<END IF>>

' <<IF KERNEL.LIBS>>

CON

' *****************************************************************************
' *                                                                           *
' *     Libary routines used                                                  *
' *                                                                           *
' *****************************************************************************

DAT

' <<INSERT LIBPTRTABLE>>

LibPtrTable     long    @LIB_MPY_I32
                long    @LIB_MPY_S32
                long    @LIB_MBG_I32
                long    @LIB_MBG_S32
                long    @LIB_DIV_I32
                long    @LIB_DIV_S32
                long    @LIB_MOD_I32
                long    @LIB_MOD_S32
                long    @LIB_SQR_S32

                long    @LIB_WaitCnt + 1
                long    @LIB_WaitPne + 3
                long    @LIB_WaitPeq + 3

                long    @LIB_TaskId
                long    @LIB_TaskStop + 1
                long    @LIB_TaskNew  + 3
                long    @LIB_TaskInit + 3

                long    @LIB_LockNew
                long    @LIB_LockSet + 1
                long    @LIB_LockClr + 1
                long    @LIB_LockRet + 1

                long    @LIB_TxByte  + 3  ' TxByte( pin, baud, data ) -Ve baud = invert signal
                long    @LIB_TxText  + 3  ' TxText( pin, baud, cPtr ) -Ve baud = invert signal
                long    @LIB_RxByte  + 2  ' RxByte( pin, baud )

' <<FINISH LIBPTRTABLE>>

' <<END IF>>

CON

' *****************************************************************************
' *                                                                           *
' *     LCC Executable Code                                                   *
' *                                                                           *
' *****************************************************************************

DAT

' <<IF DEBUG>>
' <<OR TEST>>

        long                                    ' Align to long

' <<END IF>>

' <<INSERT LCCCODE>>

LccCode byte    $85, $00, $08                   ' 0000          NUM.KM2 #MEM+main
        byte    $C0                             ' 0003          JPI
        byte    $00, $00, $00, $00              ' 0004  count:  LNG     0
        byte    $C1                             ' 0008  main:   ENT
        byte    $84, $04                        ' 0009  @1:     NUM.KM1 #MEM+count
        byte    $C2                             ' 000B          INC.I32
        byte    $9C                             ' 000C  @2:     NUM.KMZ #MEM+@1
        byte    $C0                             ' 000D          JPI
                                                ' 000E          END

' <<FINISH LCCCODE>>

CON

' *****************************************************************************
' *                                                                           *
' *     End Of LCC Executable Code                                            *
' *                                                                           *
' *****************************************************************************

