#include "p2_screenPrint.h"

#include <stdio.h>   // for printfs
#include <stdint.h>
#include <math.h>

/**
 * --- FFT bin frequency convention ---
 *
 * When performing an N-point FFT of a signal sampled at fs [Hz], the k-th FFT bin
 * corresponds to the complex amplitude of a sinusoid at the *center* frequency:
 *
 *      f_k = (k * fs) / N   [Hz],   k = 0 ... N-1
 *
 * The FFT does NOT define frequency bands or bin edges.
 * Each bin is associated with a single frequency f_k, not with a range of
 * frequencies around it.
 *
 * Precisely, the k-th FFT bin measures the projection (correlation) of the signal
 * onto a sinusoid at exactly f_k.
 *
 * Due to the finite window length (and windowing), energy from frequencies near
 * f_k may also contribute (spectral leakage). The maximum response is always at f_k.
 *
 * For real-valued signals, only bins 0 ... N/2 are unique:
 *   k = 0       -> DC (0 Hz)
 *   k = N/2     -> Nyquist (fs/2)
 *   k > N/2     -> negative frequencies: f_k = (k - N) * fs / N
 *
 * In this project, the k-th bin is nominally associated with the frequency f_k.
 *
 * --- Log / quarter-octave band aggregation ---
 *
 * The FFT provides complex coefficients X[k], k = 0 ... N-1, where X[k]
 * represents the complex amplitude of the sinusoidal component at
 * frequency f_k = (k * fs) / N.
 *
 * To obtain a reduced set of spectral values on a logarithmic frequency
 * scale (e.g. quarter-octave bands), a set of band center frequencies
 * f_1, f_2, ..., f_n is defined.
 *
 * Each band i is associated with a frequency interval determined by the
 * geometric means of adjacent centers:
 *
 *      Band_i = [ sqrt(f_{i-1} * f_i),  sqrt(f_i * f_{i+1}) )
 *
 * For the lowest band, the minimum frequency is defined as 0 Hz:
 *
 *      Band_1 = [ 0,  sqrt(f_1 * f_2) )
 *
 * The highest band is clamped to the Nyquist frequency fs/2.
 *
 * FFT bins are assigned to band i if their center frequency f_k lies
 * within the corresponding interval.
 *
 * The band magnitude is obtained by aggregating FFT bin magnitudes.
 * Two aggregation modes are supported:
 *
 *   Power-preserving aggregation
 *
 *       P_i = (1 / M_i) * sum_{k in Band_i} |X[k]|^2
 *       Level_i = 10 * log10(P_i)
 *
 *   Amplitude-preserving aggregation
 *
 *       A_i = sqrt( (1 / M_i) * sum_{k in Band_i} |X[k]|^2 )
 *       Level_i = 20 * log10(A_i)
 *
 * where M_i is the number of FFT bins assigned to band i.
 *
 * This procedure produces a constant-relative-bandwidth (constant-Q)
 * spectral representation suitable for logarithmic plots and
 * quarter-octave equaliser displays.
 */

/**
 * Generates logarithmically spaced frequency values for audio visualization bars.
 * 
 * This function creates an array of frequency values distributed logarithmically
 * from 20 Hz to approximately 20 kHz (or lower depending on parameters), which
 * corresponds to the musical octave range. The frequencies are then converted to
 * geometric means between consecutive values for smoother visualization.
 * 
 * The algorithm:
 * 1. Calculates logarithmically spaced frequencies using base-2 exponential scaling
 * 2. Computes geometric mean values between consecutive frequency points
 * 3. Applies to the final boundary using the last computed frequency
 * 
 * @param output_dim The number of frequency bars to generate. Used to determine
 *                   the frequency spacing (typically 32 for octave-based scaling).
 *                   Must be > 1 to avoid undefined behavior.
 * @param bars       Pointer to an array of int16_t values where the computed
 *                   frequency bar values will be stored. Must be allocated with
 *                   at least output_dim elements.
 * 
 * @note The maximum frequency is approximately 20480 Hz with only output_dim,
 *       and is reduced to ~13000-15000 Hz range with the jump adjustment,
 *       which is optimal for music visualization.
 * @note Frequencies are in Hertz (Hz).
 */
void generate_bars(int16_t output_dim, int16_t *bars) {
    //Function to generate the frequency associated to logarithmically spaces frequences from 20 to a value before 20480 Hz, depending on how many jumps we add/subtract
    int16_t jump = output_dim / 32;                 //32 is the base value that we consider beacuse of the octaves from 20 Hz to 20 kHz there are 31 octaves
    float incr = (float)(output_dim + jump) / 10;   //With only output dim the maximum frequency is 20480, with  + jump - 1 is around 16000 Hz, with just +jump is between 13000 Hz and 15000 Hz 
                                                    //Just +jump is probably the best for music visualisation since they tend to cut around 16 kHz

    bars[0] = 20;
    int16_t temp_value = 0;
	int16_t last = 0;

    for (int16_t i = 1; i < output_dim; i++)
    {
        temp_value = (int16_t)(20 * pow(2, i/incr) + 0.5);
		bars[i] = temp_value;
    }  
	
	temp_value = (int16_t)(20 * pow(2, output_dim/incr) + 0.5);
	last = temp_value;	
    
    //After generating the bars as above, we associate to each one of them the mid point value 
	for (int16_t i = 0; i < output_dim-1; i++) {
		bars[i] = sqrt(bars[i]*bars[i+1]);
	}
	bars[output_dim-1] = sqrt(bars[output_dim-1]*last);
}

/**
 * Generates logarithmically spaced frequency values for audio visualization bars.
 * 
 * This function creates an array of frequency values distributed logarithmically
 * from 20 Hz to approximately 20 kHz (or lower depending on parameters), which
 * corresponds to the musical octave range. The frequencies are then converted to
 * geometric means between consecutive values for smoother visualization.
 * 
 * In this case, each bin is ensured to have an associated width that is at least
 * equal to the difference between consecutive frequencies in the FFT output,
 * so that every bin can capture at least one frequency from the FFT output.
 * 
 * The algorithm:
 * 1. Calculates logarithmically spaced frequencies using base-2 exponential scaling
 * 2. Computes geometric mean values between consecutive frequency points
 * 3. Applies to the final boundary using the last computed frequency
 * 
 * @param output_dim The number of frequency bars to generate. Used to determine
 *                   the frequency spacing (typically 32 for octave-based scaling).
 *                   Must be > 1 to avoid undefined behavior.
 * @param bars       Pointer to an array of int16_t values where the computed
 *                   frequency bar values will be stored. Must be allocated with
 *                   at least output_dim elements.
 * @param freq       The frequency resolution (bin spacing) of the FFT output, 
 *                   calculated as the difference between consecutive frequency bins
 * 
 * @note The maximum frequency is approximately 20480 Hz with only output_dim,
 *       and is reduced to ~13000-15000 Hz range with the jump adjustment,
 *       which is optimal for music visualization.
 * @note Frequencies are in Hertz (Hz).
 */
void generate_bars_baseFreq(int16_t output_dim, int16_t *bars, int16_t freq) {
    //Function to generate the frequency associated to logarithmically spaces frequences from 20 to a value before 20480 Hz, depending on how many jumps we add/subtract
    int16_t jump = output_dim / 32;                 //32 is the base value that we consider beacuse of the octaves from 20 Hz to 20 kHz there are 31 octaves
    float incr = (float)(output_dim + jump) / 10;   //With only output dim the maximum frequency is 20480, with  + jump - 1 is around 16000 Hz, with just +jump is between 13000 Hz and 15000 Hz 
                                                    //Just +jump is probably the best for music visualisation since they tend to cut around 16 kHz

    bars[0] = 20;
    int16_t temp_value = 0;
	int16_t last = 0;

    for (int16_t i = 1; i < output_dim; i++)
    {
        temp_value = (int16_t)(20 * pow(2, i/incr) + 0.5);
        if (temp_value - bars[i-1] < freq)
        {
            bars[i] = bars[i-1] + freq;
        }
        else
        {
            bars[i] = temp_value;
        }
    }  
	
	temp_value = (int16_t)(20 * pow(2, output_dim/incr) + 0.5);
	if (temp_value - bars[output_dim-1] < freq)
	{
		last = bars[output_dim-1] + freq;
	}
	else
	{
		last = temp_value;
	}	
    
    //After generating the bars as above, we associate to each one of them the mid point value 
	for (int16_t i = 0; i < output_dim-1; i++) {
		bars[i] = (bars[i] + bars[i+1]) / 2;
	}
	bars[output_dim-1] = (bars[output_dim-1] + last) / 2;
}

 const int16_t octaves[32] = {	20,     25,     32,     
								40,     50,     63,     
								80,     100,    125,    
								160,    200,    250,
								315,    400,    500,    
								630,    800,    1000,
								1250,   1600,   2000,
								2500,   3150,   4000,
								5000,   6300,   8000,
								10000,  12500,  16000,
								20000, 	25000
};
/**
 * The vector octaves_print contains the values to be assigned to the output
 * frequency vector for correct octave representation in spectral analysis.
 * 
 * These values are calculated as geometric means between adjacent octave
 * frequencies, providing smooth logarithmic spacing suitable for audio
 * visualization.
 * 
 * In this representation, the first bin encompasses frequencies from 0 to 22 Hz.
 */ 
const int16_t octaves_print[31] = {	22,     28,     36,     
									45,     56,     71,     
									89,     112,    141,    
									179,    224,    281,
									355,    447,    561,    
									710,    894,    1118,
									1414,   1789,   2236,
									2806,   3550,   4472,
									5612,   7099,   8944,
									11180,  14142,  17889,
									22050
};

/**
 * @brief Generates octave-based frequency bars for audio visualization.
 * 
 * This function populates a bars array with octave frequency values for display.
 * It validates that the output dimension is exactly 31 (required for octave representation).
 * If validation fails, it falls back to generating standard bars.
 * 
 * @param output_dim The number of frequency bars to generate. Must be 31 for octave mode.
 * @param bars Pointer to an array of int16_t where the generated octave bar values will be stored.
 *             The array must be pre-allocated with at least output_dim elements.
 * 
 * @return void
 * 
 * @note If output_dim is not 31, an error message is printed and standard bars are generated instead.
 * @note The octaves_print[] array must be properly initialized before calling this function.
 * 
 * @warning The bars array must be allocated by the caller before calling this function.
 */
void generate_octaves(int16_t output_dim, int16_t *bars) {
	if (output_dim != 31) {
		printf("Error, the output dimension is not compatible with the request of representing octaves\n");
		printf("Output dimension has to be equal to 31\n");
		printf("Generating the standard output\n");
		generate_bars(output_dim, bars);
	}
	else {
		for (int16_t i = 0; i < output_dim; i++) {
			bars[i] = octaves_print[i];
		} 
	}
}


/**
 * @brief Generates an array of bar heights for a logarithmic dB scale visualization.
 *
 * @param max_input     The maximum input value to scale the output heights.
 * @param max_value     The maximum index value for the array (array size - 1).
 * @param db_reduc      The dB reduction range to apply to the scale.
 * @param bar_heights   Pointer to an array where the calculated heights will be stored.
 *                      Must be allocated with at least (max_value + 1) elements.
 *
 * @details This function calculates bar heights for a dB-based visualization by:
 *          1. Normalizing the index (0 to max_value) to a 0-1 range
 *          2. Scaling it by db_reduc and offsetting by -db_reduc
 *          3. Converting the dB value to linear scale using 10^(dB/20)
 *          4. Scaling the result by max_input
 *
 * @note The function assumes bar_heights is pre-allocated and can hold at least
 *       (max_value + 1) elements. No bounds checking is performed.
 *
 * @return void
 */
void generate_height(int16_t max_input, int16_t max_value, int16_t db_reduc, int16_t *bar_heights) {
	
	float float_value = 0.0f;
	for (int16_t i = 0; i < max_value + 1; i++)
    {
		float_value = ((((float)i / (float)max_value) * db_reduc) - db_reduc);
        bar_heights[i] = max_input * pow(10,  float_value / 20);
    }
}

/**
 * @brief Performs a binary search to find the insertion point for a data value in a sorted array of bar heights.
 *
 * @param data The value to search for in the bar heights array.
 * @param max_idx The maximum index of the bar_heights array (exclusive upper bound for search range).
 * @param bar_heights Pointer to a sorted array of bar height values to search within.
 *
 * @return int8_t The index position where data should be inserted. Returns the smallest index i
 *         where data < bar_heights[i + 1], or max_idx if data is greater than or equal to all values.
 *
 * @note This function assumes bar_heights is sorted in ascending order and has at least max_idx + 1 elements.
 * @note The search range is [0, max_idx] inclusive.
 */
int8_t bar_search(int32_t data, int16_t max_idx, const int16_t *bar_heights) {
    int8_t low = 0;
    int8_t high = max_idx;   // we search i in [0, max_idx]

    while (low < high) {
        int16_t mid = low + ((high - low) >> 1);

        // Compare with bar_heights[mid + 1]
        if (data < bar_heights[mid + 1]) {
            // i could be mid or earlier
            high = mid;
        } else {
            // i must be > mid
            low = mid + 1;
        }
    }

    return low;  // == high
}

/**
 * Input samples are signed int16_t in the range [-32768, 32767],
 * so the peak (full-scale) time-domain amplitude is A = 32767.
 *
 * In an unnormalized N-point DFT, the magnitude of a frequency bin
 * can grow by up to a factor N (|X[k]| <= N*A in the worst case).
 * For a real sinusoid exactly on an FFT bin, the magnitude is N*A/2.
 *
 * To prevent overflow in fixed-point arithmetic, each FFT butterfly
 * scales its sum/difference by >>1 (division by 2). Over log2(N)
 * stages, this produces a total normalization factor of 1/N.
 *
 * As a result, the FFT output is fully normalized: the N*A/2 growth
 * is canceled by the 1/N scaling, and a full-scale sine of amplitude A
 * produces a maximum FFT bin magnitude of A/2.
 * 
 * Applying a window w[n] multiplies the signal before the FFT and
 * reduces the FFT bin magnitude by the window coherent gain:
 *   G = (1/N) * sum_n w[n].
 *
 * With per-stage >>1 scaling (total 1/N normalization), a real
 * sinusoid of amplitude A produces a bin magnitude:
 *   |X[k]| = (A/2) * G.
 *
 * To recover the original amplitude, scale by 2/G.
 *
 * Window coherent gain G = (1/N) * sum_n w[n]
 * (used to correct FFT amplitude after windowing)
 *
 * Window type      | Coherent gain G
 * -----------------|-----------------
 * Rectangular      | 1.000
 * Hann             | 0.500
 * Hamming          | 0.540
 * Flat-top         | 0.215
 *
 */

/**
 * @brief Generates audio visualization output by mapping FFT power data to display bars based on frequency ranges.
 *
 * This function processes FFT bin data to generate the output for each
 * output bar, where each bar represents a frequency range. The bars are mapped to the
 * display using a height lookup based on the maximum power value in that frequency range.
 *
 * @param data_dim       Number of FFT bins in the input data array
 * @param freq           Bin spacing in Hz (Δf = fs/N), representing the frequency width of each bin
 * @param data           Pointer to array of int16_t containing per-bin power values (|X[k]|^2),
 *                       ordered by increasing bin frequency
 * @param output_dim     Number of output bars (display resolution)
 * @param bar_values     Pointer to array of int16_t containing upper edge frequencies (Hz) for each bar,
 *                       values must be strictly increasing
 * @param max_bar_value  Maximum power value used for scaling bar heights
 * @param bar_heights    Pointer to array of int16_t containing height lookup values for bar rendering
 * @param output         Pointer to output array of int8_t where computed bar heights are stored
 *
 * @note Uses half-open interval convention [low, high) for bin frequency mapping.
 *       Remaining bars beyond the last FFT bin are filled with zero values.
 *
 * @return void
 * 
 * The three functions below preserve different properties of the data:
 * - generate_output_amp_max: the value used to calculate the height of a bar is the maximum amplitude
 *      of the bins belonging to this frequency range
 *
 * - generate_output_amp_mean: the value used to calculate the height of a bar is the mean of the amplitudes
 *      It is the Amplitude-preserving aggregation
 * - generate_output_rms_mean: the value used to calculate the height of a bar is the square root of the mean
 *      of the amplitudes squared (RMS value). It is the Power-preserving aggregation
 * 
 */
void generate_output_amp_max(int16_t data_dim, int16_t freq, int16_t *data, int16_t output_dim, int16_t *bar_values, int16_t max_bar_value, int16_t *bar_heights, int8_t *output) {
    // Assumptions:
    // - data[i] is per-bin power (|X[k]|^2) for increasing bin frequency.
    // - freq is bin spacing in Hz (Δf = fs/N) as integer.
    // - bar_values[j] is the *upper edge* frequency (Hz) for bar j, strictly increasing.

    int16_t i = 0;
    int16_t j = 0;

    int32_t freq_bin = 0; // frequency (Hz) of current bin center associated to data[i]
	
	int32_t max = 0;

    while (j < output_dim) {

        // Accumulate bins while current bin frequency is below current bar edge.
        // Use half-open convention: [low, high)
        while (i < data_dim && freq_bin < (int32_t)bar_values[j]) {
			if (data[i] > max) {
				max = data[i];
			}
            i++;
            freq_bin += (int32_t)freq;
        }

        // Finalize current bar j
        if (max <= 0) {
            output[j] = 0;
        } else {
			output[j] = bar_search(max, max_bar_value, bar_heights);
			//printf("%d : %d\n", j, output[j]);
        }

        // Reset for next bar
        max = 0;
        j++;

        // If we ran out of FFT bins, remaining bars are zero
        if (i >= data_dim) {
            while (j < output_dim) output[j++] = 0;
            break;
        }
    }
}

void generate_output_amp_mean(int16_t data_dim, int16_t freq, int16_t *data, int16_t output_dim, int16_t *bar_values, int16_t max_bar_value, int16_t *bar_heights, int8_t *output) {
    // Assumptions:
    // - data[i] is per-bin power (|X[k]|^2) for increasing bin frequency.
    // - freq is bin spacing in Hz (Δf = fs/N) as integer.
    // - bar_values[j] is the *upper edge* frequency (Hz) for bar j, strictly increasing.

    int16_t i = 0;
    int16_t j = 0;

    int32_t acc = 0;   // accumulated power in current bar
    int16_t M   = 0;   // number of FFT bins accumulated

    int32_t freq_bin = 0; // frequency (Hz) of current bin center associated to data[i]

    while (j < output_dim) {

        // Accumulate bins while current bin frequency is below current bar edge.
        // Use half-open convention: [low, high)
        while (i < data_dim && freq_bin < (int32_t)bar_values[j]) {
            acc += (int32_t)(data[i]);
            i++;
			M++;
            freq_bin += (int32_t)freq;
        }

        // Finalize current bar j
        if (M <= 0 || acc <= 0) {
            output[j] = 0;
        } else {
			output[j] = bar_search(acc / M, max_bar_value, bar_heights);
			//printf("%d : %d\n", j, output[j]);
        }

        // Reset for next bar
        acc = 0;
        M   = 0;
        j++;

        // If we ran out of FFT bins, remaining bars are zero
        if (i >= data_dim) {
            while (j < output_dim) output[j++] = 0;
            break;
        }
    }
}

void generate_output_rms_mean(int16_t data_dim, int16_t freq, int16_t *data, int16_t output_dim, int16_t *bar_values, int16_t max_bar_value, int16_t *bar_heights, int8_t *output) {
    // Assumptions:
    // - data[i] is per-bin power (|X[k]|^2) for increasing bin frequency.
    // - freq is bin spacing in Hz (Δf = fs/N) as integer.
    // - bar_values[j] is the *upper edge* frequency (Hz) for bar j, strictly increasing.

    int16_t i = 0;
    int16_t j = 0;

    int32_t acc = 0;   // accumulated power in current bar
    int16_t M   = 0;   // number of FFT bins accumulated

    int32_t freq_bin = 0; // frequency (Hz) of current bin center associated to data[i]

    while (j < output_dim) {

        // Accumulate bins while current bin frequency is below current bar edge.
        // Use half-open convention: [low, high)
        while (i < data_dim && freq_bin < (int32_t)bar_values[j]) {
            acc += (int32_t)(data[i] * data[i]);
            i++;
			M++;
            freq_bin += (int32_t)freq;
        }

        // Finalize current bar j
        if (M <= 0 || acc <= 0) {
            output[j] = 0;
        } else {
			output[j] = bar_search(sqrt(acc / M), max_bar_value, bar_heights);
			//printf("%d : %d\n", j, output[j]);
        }

        // Reset for next bar
        acc = 0;
        M   = 0;
        j++;

        // If we ran out of FFT bins, remaining bars are zero
        if (i >= data_dim) {
            while (j < output_dim) output[j++] = 0;
            break;
        }
    }
}

/*
    THE FOLLOWING CODE HAS BEEN LEFT FOR RETROCOMPATIBILITY, BUT IT IS NO LONGER USED.

    Initially developed to maximize the resolution of the graphical output.
    The mathematically correct formulas used in the function above are now
    the default.
	
	This functions require more parameters that the new ones
	Using the Hann window the data that lead to the best results is the following
	max_value = 180
	rescaling_value = 4
*/

//Output formatting
void old_generate_bars_lowValue(int16_t output_dim, int16_t *bars, int16_t freq) {
    //Function to generate the frequency associated to logarithmically spaces frequences from 20 to a value before 20480 Hz, depending on how many jumps we add/subtract
    int16_t jump = output_dim / 32;                 //32 is the base value that we consider beacuse of the octaves from 20 Hz to 20 kHz there are 31 octaves
    float incr = (float)(output_dim + jump) / 10;   //With only output dim the maximum frequency is 20480, with  + jump - 1 is around 16000 Hz, with just +jump is between 13000 Hz and 15000 Hz 
                                                    //Just +jump is probably the best for music visualisation since they tend to cut around 16 kHz

    bars[0] = 20;
    int16_t temp_value = 0;

    for (int16_t i = 1; i < output_dim; i++)
    {
        temp_value = (int16_t)(20 * pow(2, i/incr) + 0.5);
        if (temp_value - bars[i-1] < freq)
        {
            bars[i] = bars[i-1] + freq;
        }
        else
        {
            bars[i] = temp_value;
        }
    }   
    
    //printf("Max frequence %6d Hz", bars[output_dim-1]);
}

void old_generate_bars_meanValue(int16_t output_dim, int16_t *bars, int16_t freq) {
    //Function to generate the frequency associated to logarithmically spaces frequences from 20 to a value before 20480 Hz, depending on how many jumps we add/subtract
    int16_t jump = output_dim / 32;                 //32 is the base value that we consider beacuse of the octaves from 20 Hz to 20 kHz there are 31 octaves
    float incr = (float)(output_dim + jump) / 10;   //With only output dim the maximum frequency is 20480, with  + jump - 1 is around 16000 Hz, with just +jump is between 13000 Hz and 15000 Hz 
                                                    //Just +jump is probably the best for music visualisation since they tend to cut around 16 kHz

    bars[0] = 20;
    int16_t temp_value = 0;
	int16_t last = 0;

    for (int16_t i = 1; i < output_dim; i++)
    {
        temp_value = (int16_t)(20 * pow(2, i/incr) + 0.5);
        if (temp_value - bars[i-1] < freq)
        {
            bars[i] = bars[i-1] + freq;
        }
        else
        {
            bars[i] = temp_value;
        }
    }  
	
	temp_value = (int16_t)(20 * pow(2, output_dim/incr) + 0.5);
	if (temp_value - bars[output_dim-1] < freq)
	{
		last = bars[output_dim-1] + freq;
	}
	else
	{
		last = temp_value;
	}	
    
    //After generating the bars as above, we associate to each one of them the mid point value 
	for (int16_t i = 0; i < output_dim-1; i++) {
		bars[i] = (bars[i] + bars[i+1]) / 2;
	}
	bars[output_dim-1] = (bars[output_dim-1] + last) / 2;
}

void old_generate_octaves_lowValue(int16_t output_dim, int16_t *bars, int16_t freq) {
	if (output_dim != 31) {
		printf("Error, the output dimension is not compatible with the request of representing octaves\n");
		printf("Output dimension has to be equal to 31\n");
		printf("Generating the standard output\n");
		old_generate_bars_lowValue(output_dim, bars, freq);
	}
	else {
		for (int16_t i = 0; i < output_dim; i++) {
			bars[i] = octaves[i];
		} 
	}
}

void old_generate_octaves_meanValue(int16_t output_dim, int16_t *bars, int16_t freq) {
	if (output_dim != 31) {
		printf("Error, the output dimension is not compatible with the request of representing octaves\n");
		printf("Output dimension has to be equal to 31\n");
		printf("Generating the standard output\n");
		old_generate_bars_meanValue(output_dim, bars, freq);
	}
	else {
		for (int16_t i = 0; i < output_dim; i++) {
			bars[i] = (octaves[i] + octaves[i + 1]) / 2;
		} 
	}
}

void old_generate_height(int16_t max_input, int16_t max_value, int16_t *bar_heights) {
    float rescaling = (float)max_input / (max_value * max_value);
    
    //By trial and error we found that most of the values are between 0 and 180
    bar_heights[0] = 0;

    for (int16_t i = 1; i < max_value; i++)
    {
        bar_heights[i] = ((i * i * rescaling) + 0.5);
    }
    
    bar_heights[max_value] = max_input;
}

int8_t old_bar_heigh(int32_t data, int16_t max_value, int16_t *bar_heights) {
    for (int16_t i = 0; i < max_value; i++) 
    {
        if (data < bar_heights[i+1]) 
            return i;
    }
    return max_value;   // default if nothing matches
}

int8_t old_bar_heigh_binary(int32_t data, int16_t max_value, const int16_t *bar_heights) {
    int8_t low = 0;
    int8_t high = max_value;   // we search i in [0, max_value]

    while (low < high) {
        int16_t mid = low + ((high - low) >> 1);

        // Compare with bar_heights[mid + 1]
        if (data < bar_heights[mid + 1]) {
            // i could be mid or earlier
            high = mid;
        } else {
            // i must be > mid
            low = mid + 1;
        }
    }

    return low;  // == high
}

void old_generate_output_max(int16_t  data_window_dim, int16_t freq, int16_t *data, int16_t output_dim, int16_t *bar_values, int16_t max_value, int16_t *bar_heights, int8_t *output, int16_t rescaling_value) {
	int16_t i = 0;
    int16_t j = 0;
    int32_t temp_output = 0;

    int16_t freq_bin = (freq >> 1);
    while (j < output_dim && i < data_window_dim)
    {
        if (freq_bin <= bar_values[j])
        {
			if (temp_output < data[i])
				temp_output = data[i];
            i++;
            freq_bin += freq;
        }
        else
        {
            if (temp_output == 0)
            {
                output[j] = 0;
            }
            else
            {
                output[j] = old_bar_heigh_binary(( temp_output >> rescaling_value ) * output_dim / (output_dim - j + 2) , max_value, bar_heights);
            }          
            temp_output = 0;
            j++;    
        }
    }

    if (j < output_dim && temp_output > 0)
    {
	    output[j] = old_bar_heigh_binary(( temp_output >> rescaling_value ) * output_dim / (output_dim - j + 2) , max_value, bar_heights);
    }
}

void old_generate_output_mean(int16_t  data_window_dim, int16_t freq, int16_t *data, int16_t output_dim, int16_t *bar_values, int16_t max_value, int16_t *bar_heights, int8_t *output, int16_t rescaling_value) {
    int16_t i = 0;
    int16_t j = 0;
    int32_t temp_output = 0;
	int32_t tot = 0;

    int16_t freq_bin = (freq >> 1);
    while (j < output_dim && i < data_window_dim)
    {
        if (freq_bin <= bar_values[j])
        {
			temp_output += data[i];
            i++;
            freq_bin += freq;
			tot ++;
        }
        else
        {
            if (temp_output == 0)
            {
                output[j] = 0;
            }
            else
            {
                output[j] = old_bar_heigh_binary(( (temp_output / tot) >> rescaling_value ) * output_dim / (output_dim - j + 2) , max_value, bar_heights);
            }          
            temp_output = 0;
            j++;    
			tot = 0;
        }
    }
	
	if (j < output_dim && temp_output > 0)
    {
        output[j] = old_bar_heigh_binary(( (temp_output / tot) >> rescaling_value ) * output_dim / (output_dim - j + 2) , max_value, bar_heights);
    }
}
